function [x,fval,stat,out,lambda] = ipmwrapper(H,f,A,b,Aeq,beq,lb,ub,x0,opts)
% IPMWRAPPER wrapper for ipm() to get uniform call w.r.t. quadprog
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin < 10
  opts = ipmopts();
end

%% Sizes
nx0 = numel(f);
neq0 = numel(beq);
nineq0 = numel(b);
if isempty(A  ), A   = zeros(0,nx0); b   = zeros(0,1); end
if isempty(Aeq), Aeq = zeros(0,nx0); beq = zeros(0,1); end

%% Initialize outputs
fval = []; % not used
out = []; % not used
x = zeros(nx0,1);
lambda.eqlin   = zeros(neq0,1);
lambda.ineqlin = zeros(nineq0,1);
lambda.lower   = zeros(nx0,1);
lambda.upper   = zeros(nx0,1);

%% Variables directly constrained
nx = nx0;
l = (sum(Aeq ~= 0,2) == 1);       % equality involves just one x value
Cd = Aeq(l,:); bd = beq(l,1);     % take these Cd*x = bd
idirect = any(Cd,1);              % x indices to solve for directly
xdirect = bd./sum(Cd,2);          % solve
Aeq = Aeq(~l,:); beq = beq(~l,1); % eliminate these eq. constraints
% Check that some indices are not constrained more than once
if numel(x(idirect)) ~= numel(xdirect)
  keep = false(size(xdirect));
  for ii = 1:find(idirect)          % Loop over constrained indices
    mask = Cd(:,ii) ~= 0;           % Rows of Cd/xdirect corresponding to this index
    x_ = xdirect(mask);             % Values of x for this index
    % Compare values if more than one is found
    if numel(x_) > 1 && any(abs(x_(2:end)-x_(1))>10*eps)
      % If values are too different, trigger infeasible flag
      if opts.debug, fprintf('IPM found incompatible equality constraints - the problem is infeasible\n'); end
      stat = -2; return % infeasible
    else
      keep(find(mask,1)) = true;    % Keep only first
    end
  end
  if sum(~keep)
    warning('ipm:RedundantEqConstraints','eliminated %d redundant equality constraints',sum(~keep))
  end
  xdirect = xdirect(keep);          % Eliminate redundant values
end

%% Variables that are not part of the optimization problem
inosolve = ~any([H;Aeq;A],1);  % indices of x variables that are not used in the optimization problem
                               % this happens e.g. if Fb is not constrained in FBT

% Flag unsolved variables that have bounds
havebounds = ~(isempty(lb) || isempty(ub));
if havebounds
  ilb   = inosolve & ( isfinite(lb) & ~isfinite(ub))'; % unsolved variables with a lower bound
  iub   = inosolve & (~isfinite(lb) &  isfinite(ub))'; % unsolved variables with an upper bound
  iboth = inosolve & ( isfinite(lb) &  isfinite(ub))'; % unsolved variables with both bounds
  
  % save these bounds as they will be removed later
  lbb = lb(ilb); ubb = ub(iub);
  if any(iboth), bbb(:,1:2) = [lb(iboth),ub(iboth)];
  else,          bbb = zeros(0,2); % init
  end
end

% update matrices and right hand sides
islv = ~(idirect|inosolve); % indices to solve for
if ~all(islv)
  H0 = H; f0 = f; A0 = A; % Save original problem (Aeq0 not needed)
  f   = f(islv) + H(islv,idirect)*xdirect;
  H   = H(islv,islv);
  beq = beq - Aeq(:,idirect)*xdirect;
  Aeq = Aeq(:,islv);
  b   = b - A(:,idirect)*xdirect;
  A   = A(:,islv);
  if ~isempty(lb), lb = lb(islv); end
  if ~isempty(ub), ub = ub(islv); end
  nx = numel(f);
end

%% Inequality constraints
% Group bounds with constraints and flip A,b sign (quadprog has A*x<b, ipm has A*x>b)
nineq = numel(b);
nlb = numel(lb);
nub = numel(ub);
A = [-A;eye(nlb);-eye(nub)];
b = [-b;lb;-ub];

%% Equality constraints checks 
% Check for linearly dependent equality constraints
neq = numel(beq);
ikeep = true(neq,1);
if neq
  % Identify which rows of Aeq can be eliminated, QR operates on the
  % columns of the input matrix, so we use the matrix transpose.
  if opts.presolve
    % Extra columns of Q make up the Null space of Aeq which we will need later
    [Q,R,P] = qr(Aeq.','vector');
  else
    % Use the economic decomposition
    [Q,R,P] = qr(Aeq.',0);
  end
  % Aeq(:,P).' = Q*R and diag(R) is sorted by absolute values
  k = find(abs(diag(R))>10*eps,1,'last'); % Rows with 0 on the diagonal can be eliminated
  if isempty(k), k = 0; end
  % R = [R11,R12;0,R22] with R22~0 and R11 of size [k,k]

  if k < neq
    % check that b for rows to be eliminated are indeed compatible
    beqP = beq(P,1); % Reorder be rows as Aeq
    resbe = beqP(k+1:end,1) - (R(1:k,1:k)\R(1:k,k+1:end)).'*beqP(1:k,1);
    if any(abs(resbe)>10*eps)
      if opts.debug, fprintf('IPM found incompatible equality constraints - the problem is infeasible\n'); end
      stat = -2; return % infeasible
    else
      warning('ipm:RedundantEqConstraints','eliminated %d redundant equality constraints',neq-k)
      ikeep(P(k+1:end)) = false; % Mark rows for deletion
      Aeq = Aeq(ikeep,:); beq = beq(ikeep,1);
    end
  end
end

%% Presolve equality constraints
% Eliminate equality constraints
dopresolve = ~isempty(Aeq) & opts.presolve;
if dopresolve
  % Save original problem
  Aeq_ = Aeq;
  H_ = H;
  f_ = f;
  A_ = A;
  % Do the elimination
  Ne = Q(:,k+1:end);
  xo = Aeq\beq;                  % x = N*v + xo; Ax < b => A*N*v < b - A*xo
  f  = (Ne'*(H'*xo+f));          % 0.5*x'*H*x + x'*f = 0.5*(N*v+xo)'*H*(N*v+xo) + (v'*N' + xo')*f;
  H  = Ne'*H*Ne;                 % = v'*(N'*H*N)*v + v'*(N'*H*xo + N'*f) + constants
  b  = b - A*xo;
  A  = A*Ne;
  nx  = numel(f);
  Aeq = zeros(0,nx); beq = zeros(0,1);
  % Note if bounds were still present we would have to transform them to inequalities
end

%% Eliminate empty or infinite inequality constraints
ioki = any(A,2) & isfinite(b); 
if any(b(~ioki)>0), error('0*x>=b constraint with b>0'); end
A = A(ioki,:); b=b(ioki,:);

%% switch ipm mode depending on situation - see ipm help
if isempty(beq) && ~isempty(b)
  % inequality constraints with no equality constraints
  if ~any(b) && isequal(A,eye(nx))
    % only inequality constraints: X>0
    ipmmode = 3;
  elseif ~any(b) && isequal(A/A(1,1),eye(nx))
    % only inequality constraints: A*X>0
    ipmmode = 4;
  else
    [i,j,s] = find(A);
    if (numel(i) == 1 && b(i) == 0)
      % only inequality constraints: A*x(b)>0
      ipmmode = 5; A = s; b = j;
    else
      % only inequality constraints: general case
      ipmmode = 2;
    end
  end
elseif isempty(beq)
  % No inequality constraints, no equality constraints
  ipmmode = -1; % Direct solver
elseif opts.useoptimized
  % General case, use optimized implementation
  ipmmode = 1;
else
  % General case, non optimized implementation
  ipmmode = 0;
end

if ipmmode>=0
  [x,y,~,z,~,stat] = ipm(H,f,A,b,Aeq,beq,x0,[],[],[],ipmmode,opts.niter,opts.tol,opts.debug);
else
  % No constraints - direct solver
  x = -H\f;
  y = zeros(0,1);
  z = zeros(0,1);
  stat = 1;
  if opts.debug
    fprintf('ipmwrapper: direct solve without constraints\n');
  end
end

%% post solve
if stat >= 0 % IPM has not errored
  % Empty or infinite inequality constraints
  z_ = zeros(numel(ioki),1);
  z_(ioki) = z;
  z = z_;

  % Presolve equality constraints
  if dopresolve
    x = Ne*x + xo;
    y = - (Aeq_.'\(H_*x + f_ - A_.'*z));
  end

  % linearly dependent equality constraints
  y_ = zeros(neq,1);
  y_(ikeep) = y;
  y = y_;

  % Group bounds with constraints and flip A,b sign
  [kineq,klb,kub] = n2k(nineq,nlb,nub);
  zin = z(kineq);
  if nlb, zlb = z(klb); else, zlb = zeros(numel(x),1); end
  if nub, zub = z(kub); else, zub = zeros(numel(x),1); end

  % Variables directly constrained / Variables that are not part of the optimization problem
  if ~all(islv)
    %  - Unknown
    x_ = zeros(nx0,1);
    x_(idirect) = xdirect;
    x_(islv) = x;
    % treat non-solved variables that have bounds
    if havebounds
      x_(ilb) = max(lbb,0); % lower bound only, zero if possible
      x_(iub) = min(ubb,0); % upper bound only, zero if possible
      izero = bbb(:,2)>=0 & bbb(:,1)<=0 ; % bounds envelop zero
      x_(iboth &  izero) = 0; % assign zero if possible
      x_(iboth & ~izero) = mean(bbb(~izero,:),2); % mean otherwise
    end
    % assign x
    x = x_;
    %  - lower bound lagrange multiplier
    zlb_ = zeros(nx0,1);
    zlb_(islv) = zlb;
    zlb = zlb_;
    %  - upper bound lagrange multiplier
    zub_ = zeros(nx0,1);
    zub_(islv) = zub;
    zub = zub_;
    %  - equality constraint lagrange multiplier
    y_ = zeros(neq0,1);
    y_(~l) = y;
    rd = H0*x + f0 + A0.'*zin + zub - zlb;
    % Handle duplicate direct constraints
    if numel(xdirect) < size(Cd,1)
      y__ = zeros(size(Cd,1),1);
      y__(keep) = -rd(idirect)./sum(Cd(keep,:),2);
    else
      y__ = -rd(idirect)./sum(Cd,2);
    end
    y_( l) = y__;
    y = y_;
  end
  % Store lagrange coefficients in structure
  lambda.eqlin   = y;
  lambda.ineqlin = zin;
  lambda.lower   = zlb;
  lambda.upper   = zub;
else
  x = zeros(nx0,1);
end

end
