%IPM  Interior point method general minimisation
% [X,Y,S,Z,KT,STAT] = IPM(H,C,A,B,AE,BE,XI,YI,SI,ZI,MODE[,NIT,TOL,VERB])
% finds X such that min X'*H*X/2+C'*X, A*X>B, AE*X=BE using the interior
% point method. XI,YI,SI,ZI are initial guesses. NIT and TOL control the
% iteration numbers and the convergence criterion. VERB sets the verbosity
% MODE =
% .0 for the brute force implementation
% .1 for the optimised implementation
% .2 no equality constraints min X'*H*X/2+C'*X, A*X>B
% .3 min X'*H*X/2+C'*X, X>0
% .4 min X'*H*X/2+C'*X, A(1)*X>0
% .5 min X'*H*X/2+C'*X, A(1)*X(B(1))>0
%
% STAT: exit flag, same meaning as in QUADPROG
%  1: Function converged to the solution x.
%  0: Number of iterations exceeded NIT
% -2: Infeasible due to conflicting equality constraints
%
% For details, see: [MEQ-redbook] 
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Not yet implemented errors (copied from quadprog help):
% -2: The step size was smaller than the step tolerance, but constraints were not satisfied (infeasible)
% -3: Problem is unbounded.
%  2: Step size was smaller than the step tolerance, constraints were satisfied.
% -6: Nonconvex problem detected.
% -8: Unable to compute a step direction.

function [x,y,s,z,kt,stat] = ipm(H,c,A,b,Ae,be,x,y,s,z,mode,niter,tol,verbosity)
 
 if nargin < 12 || isempty(niter)     , niter = 100;        end
 if nargin < 13 || isempty(tol)       , tol = 1e-6;         end
 if nargin < 14 || isempty(verbosity) , verbosity = 0;      end

 kt = 0; stat = 0; % default status flag
 n  = size(H ,1);
 if isempty(A ), A  = zeros(0,n); b  = zeros(0,1); end
 if isempty(Ae), Ae = zeros(0,n); be = zeros(0,1); end

 %% Checks
 % Check basic sizes
 assert(all(size(c)==[n,1]),'wrong size for c, expected [%d %d]',n,1);
 assert(size(H,2)==n,'H must be square');
 
 %% init
 ni = size(A ,1); ne = size(Ae,1);
 if isempty(x), x = ones(n ,1); end
 if isempty(y), y = ones(ne,1); end
 if isempty(s), s = ones(ni,1); end
 if isempty(z), z = ones(ni,1); end

 % Without inequality constraints, use mode 0 or 1
 if ni==0 && mode>1, mode=1; end

 %% solve
 switch mode
  case 0 % min x'*H*x/2+c'*x, A*x>b, Ae*x=be
   % Initialize Newton step matrix
   D = [
        H           Ae'          zeros(n,ni)  -A'         ;
        Ae          zeros(ne,ne) zeros(ne,ni) zeros(ne,ni);
        zeros(ni,n) zeros(ni,ne) diag(z)      diag(s)     ;
        -A          zeros(ni,ne) eye(ni)      zeros(ni,ni);
       ];
   % Indices for different blocks
   [ix,iy,is,iz] = n2k(n,ne,ni,ni);
   % Iteration loop
   for kt = 1:niter
    % Predictor step
    rd = H*x + c - A'*z + Ae'*y;
    re = Ae*x - be;
    rc = z.*s;
    rp = s - A*x + b;
    D(is,is) = diag(z); D(is,iz) = diag(s);
    r = [rd ; re ; rc ; rp];
    d = - D \ r;
    ds = d(is);
    dz = d(iz);
    % Largest step ensuring s>0 and z>0
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,alphas,alphaz]);
    % Update barrier parameter
    tau = s'*z/ni;
    tau1 = (s + alpha*ds)'*(z + alpha*dz)/ni;
    sigma = (tau1/tau).^3;
    % Centering-corrector step
    r(is) = r(is) + (ds.*dz - sigma*tau);
    d = - D \ r;
    dx = d(ix);
    dy = d(iy);
    ds = d(is);
    dz = d(iz);
    % Largest step ensuring s>0 and z>0 with some margin
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,0.99*alphas,0.99*alphaz]);
    % Update
    x = x + alpha*dx;
    y = y + alpha*dy;
    s = s + alpha*ds;
    z = z + alpha*dz;
    % Exit conditions
    res  = max(abs(r));
    step = max(abs(d));
    if verbosity, ipmdebug(mode,kt,res,step,tol); end
    if res  < tol; stat=1; break, end
    if step < tol; stat=1; break, end
   end
   
  case 1 % min x'*H*x/2+c'*x, A*x>b, Ae*x=be, optimised solution
   % Iteration loop
   for kt = 1:niter
    % Predictor step
    rd = H*x + c - A'*z + Ae'*y;
    re = Ae*x - be;
    rc = z.*s;
    rp = s - A*x + b;
    d = z./s;
    [D ,flag] = chol(H + A'*diag(d)*A);
    if flag, break; end
    [De,flag] = chol(Ae*(D\(D'\Ae'))); % See eq. (5.16) in MEQ-Redbook
    if flag, break; end
    xc = -rc./z;
    xp = d.*(rp+xc);
    xd = A'*xp-rd;
    xe = re+Ae*(D\(D'\xd));
    dy = De\(De'\xe);
    dx = D\(D'\(xd-Ae'*dy));
    dz = xp - d.*(A*dx);
    ds = xc-s./z.*dz;
    % Largest step ensuring s>0 and z>0
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,alphas,alphaz]);
    % Update barrier parameter
    tau = s'*z/ni;
    tau1 = (s + alpha*ds)'*(z + alpha*dz)/ni;
    sigma = (tau1/tau).^3;
    % Centering-corrector step
    rc = z.*s+(ds.*dz-sigma*tau);
    xc = -rc./z;
    xp = d.*(rp+xc);
    xd = A'*xp-rd;
    xe = re+Ae*(D\(D'\xd));
    dy = De\(De'\xe);
    dx = D\(D'\(xd-Ae'*dy));
    dz = xp - d.*(A*dx);
    ds = xc-s./z.*dz;
    % Largest step ensuring s>0 and z>0 with some margin
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,0.99*alphas,0.99*alphaz]);
    % Update
    x = x + alpha*dx;
    y = y + alpha*dy;
    s = s + alpha*ds;
    z = z + alpha*dz;
    % Exit conditions
    res  = max(abs([rd;re;rc;rp]));
    step = max(abs([dx;dy;ds;dz]));
    if verbosity, ipmdebug(mode,kt,res,step,tol); end
    if res  < tol; stat=1; break, end
    if step < tol; stat=1; break, end
   end
   
  case 2 % min x'*H*x/2+c'*x, A*x>b
    UH = H(triu(true(size(H))));
    [x,s,z,kt,stat] = ipm2mex(UH,c,A,b,x,s,z,tol,niter,logical(verbosity));
    stat = double(stat);
   
  case 3 % min x'*H*x/2+c'*x, x>0
   UH = H(triu(true(size(H))));
   [x,z,kt,stat] = ipm4mex(UH,c,1,x,z,tol,niter,logical(verbosity));
   s = x;
   stat = double(stat);

  case 4 % min x'*H*x/2+c'*x, a*x>0
   UH = H(triu(true(size(H))));
   a = A(1,1);
   [x,z,kt,stat] = ipm4mex(UH,c,a,x,z,tol,niter,logical(verbosity));
   % Rescale s and z
   s = x*a;
   z = z/a;
   stat = double(stat);

  case 5 % min x'*H*x/2+c'*x, a*x(b)>0
   % Rescale so that A=+/-1
   a = abs(A);
   A = A/a;
   s = max(1,A*x(b)); x(b) = s/A;
   % Iteration loop
   for kt = 1:niter
    % Predictor step
    rd = H*x + c; rd(b) = rd(b) - A*z;
    rc = z.*s;
    d = z./s;
    D = H; D(b,b) = D(b,b) + d;
    [D,flag] = chol(D);
    if flag, break; end
    xc = -rc./s;
    xd = -rd; xd(b) = xd(b) + A*xc;
    dx = D\(D'\xd);
    ds = A*dx(b);
    dz = xc - d.*ds;
    % Largest step ensuring s>0 and z>0
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,alphas,alphaz]);
    % Update barrier parameter
    tau = s*z/ni;
    tau1 = (s + alpha*ds)*(z + alpha*dz)/ni;
    sigma = (tau1/tau).^3;
    % Centering-corrector step
    rc = rc+(ds.*dz-sigma*tau);
    xc = -rc./s;
    xd = -rd; xd(b) = xd(b) + A*xc;
    dx = D\(D'\xd);
    ds = A*dx(b);
    dz = xc - d.*ds;
    % Largest step ensuring s>0 and z>0 with some margin
    k = (ds < 0); alphas = min(-s(k)./ds(k));
    k = (dz < 0); alphaz = min(-z(k)./dz(k));
    alpha = min([1,0.99*alphas,0.99*alphaz]);
    % Update
    z = z + alpha*dz;
    x = x + alpha*dx;
    s = A*x(b);
    % Exit conditions
    res  = max(abs([rd;rc]));
    step = max(abs([dx;dz]));
    if verbosity, ipmdebug(mode,kt,res,step,tol); end
    if res  < tol; stat=1; break, end
    if step < tol; stat=1; break, end
   end
   % Rescale s and z for original problem
   s = s*a;
   z = z/a;
 end

 % optional output displays
 if verbosity
   switch stat
     case  0, fprintf('IPM mode %d exceeded number of iterations (%d)\n'    ,mode,niter);
     case  1, fprintf('IPM mode %d converged to tol=%5.0e in %d iterations\n',mode,tol,kt);
   end
 end
end

function ipmdebug(mode,kt,res,step,tol)
fprintf('ipm-%d it=%3d res=%5.2e step=%5.2e tol=%5.2e\n',mode,kt,res,step,tol)
end
