classdef vveig_test < genlib_test
  % Tests for vveig
  %
  % [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    rtol     = 1.e-3; %tolerance on the elements relative difference 
    atol     = 1.e-4; %tolerance on the elements absolute difference
    nv       = 256;    %number of matrix elements
    shot     = 61400;
    
    Mvv,Rv
  end
  
  methods (TestMethodSetup)
    function setup_matrices(testCase)
      %Setup Mvv and Rv in order to be symmetric and positive definite (Rv
      %is diagonal and represented by an array)
      testCase.Mvv = rand(testCase.nv);
      testCase.Mvv = testCase.Mvv*testCase.Mvv';
      testCase.Rv  = rand(testCase.nv,1);
      testCase.Rv  = testCase.Rv.*testCase.Rv;
    end
  end
  
  methods (Test)   
    function test_vveig(testCase)
        %Test diagonalisation with type='R'
        [Tve,Le,Re] = vveig(testCase.Mvv,testCase.Rv,'R');
        testCase.verifyEqual(diag(Le),Tve'*testCase.Mvv*Tve,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Le from vveig is different than Tve''*Mvv*Tve for type=R');
        testCase.verifyEqual(diag(Re*ones(size(Le))),Tve'*diag(testCase.Rv)*Tve,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Re from vveig is different than Tve''*diag(Rv)*Tve for type=R');
        testCase.verifyEqual(esort(eig(diag(testCase.Rv)\testCase.Mvv)),Le./Re,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Le./Re from vveig is different than the eigenvalues of diag(Rv)\Mvv for type=R');
          
        %Test diagonalisation with type='M'
        [Tve,Le,Re] = vveig(testCase.Mvv,testCase.Rv,'M');
        testCase.verifyEqual(diag(Le*ones(size(Re))),Tve'*testCase.Mvv*Tve,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Le from vveig is different than Tve''*Mvv*Tve for type=M');
        testCase.verifyEqual(diag(Re),Tve'*diag(testCase.Rv)*Tve,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Re from vveig is different than Tve''*diag(Rv)*Tve for type=M');
        testCase.verifyEqual(flipud(esort(eig(testCase.Mvv\diag(testCase.Rv)))),Re./Le,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Le./Re from vveig is different than the eigenvalues of diag(Rv)\Mvv for type=M');  
    end
    
    function test_vveig_static(testCase)
      % Compare results with stored values in TCV static tree
      check = ~isempty(which('mdsipmex')); % Check that SPC MDSplus tools are available
      check = check && (mdsconnect('tcvdata.epfl.ch')>0); % Check that connection to TCV MDSplus server is possible
      testCase.assumeTrue(check,'Skipping comparison of vveig with TCV static tree since unable to connect to TCV MDSplus server');
      mdsopen('static',testCase.shot);
      Mvv_static = mdsvalue('static("mut_v_v")');
      Rv_static = mdsvalue('static("res_v_v")');
      [~,Le,Re] = vveig(Mvv_static,diag(Rv_static));
      Muu = mdsvalue('static("mut_e_e")');
      Ru = mdsvalue('static("res_e_e")');
      testCase.verifyEqual(diag(Le),Muu,...
            'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Muu from vveig is different from the one stored in the static tree');
      testCase.verifyEqual(Re*ones(size(Ru)),Ru,...
          'AbsTol',testCase.atol,'RelTol',testCase.rtol,'Ru from vveig is different from the one stored in the static tree');
    end
  end
end