classdef iflooriceiliround_test < genlib_test
  % Tests for ifloor and iround
  %
  % [+GenLib General Purpose Library+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties(TestParameter)
    transposeX = {false,true};
    transposeT = {false,true};
    scalar     = {false,true};
    downselect = {'first','last'};
    fun = {'iceil','ifloor','iround'}
  end
  
  methods (Test)
    
    function test_iflooriceiliroundmex(testCase,fun,transposeX,transposeT,downselect)
      W = 0;
      
      % Verify ceil behaviour
      T = [1 2 3];
      X = [0 1 2.4 2.5 2.6 3 4];
      
      switch downselect
        case 'first', X=X(1);
        case 'last', X=X(end);
        otherwise % no change
      end
      
      if transposeX
        X = X';
      end
      if transposeT
        T = T';
      end
      
      switch fun
        case 'iround'
          expfun = @iround_exp; % defined below
        case 'ifloor'
          expfun = @ifloor_exp;
        case 'iceil'
          expfun = @iceil_exp;
      end
      
      expected = expfun(T,X);
      
      func = str2func(fun);
      testCase.assertEqual(func(T,X),expected);
      
      % Verify iflooriceiliroundmex errors when empty table is used
    end
    
    function test_nonsorted(testCase,fun)
      T = [0 1 2];
      X = [2 1];
      ifunc = str2func(fun);
      Expected = [3,2];
      
      % Test that unsorted does not give right result
      testCase.verifyNotEqual(ifunc(T,X),Expected)
      % Check that third argument triggers sorting
      testCase.verifyEqual(ifunc(T,X,1),Expected)
    end
    
    function test_empty(testCase)
      % empty table should throw error
      T = []; W = int32(0); X = [0 1 2.4 2.5 2.6 3 4];
      testCase.verifyError(@() iflooriceiliroundmex(T,X,W),'iflooriceiliroundmex:emptyT');
      
      % empty X should return empty result
      T = 1; W = int32(0); X = [];
      testCase.verifyEqual([],iflooriceiliroundmex(T,X,W));
    end
  end
end

%% aux functions with expected behaviour

function ii=iround_exp(T,X)
% expected iround behaviour
ii = zeros(size(X)); % init
for jj=1:numel(X)
  [~,ii(jj)] = min(abs(X(jj)-T));
end
end

function ii=iceil_exp(T,X)
% expected iceil behaviour
ii = zeros(size(X)); % init

for jj=1:numel(X)
  kk = find(T>=X(jj),1,'first');
  if ~isempty(kk), ii(jj)=kk;
  else, ii(jj)=0; % error flag
  end
end
end

function ii=ifloor_exp(T,X)
% expected ifloor behaviour
ii = zeros(size(X)); % init

for jj=1:numel(X)
  kk = find(T<=X(jj),1,'last');
  if ~isempty(kk), ii(jj)=kk;
  else, ii(jj)=0; % error flag
  end
end
end
