function [S,K,U,Udot,maskL,dxdxL,dxdu] = fgselim(L,Jx,Ju,Jxdot)
% FGSELIM compute the reduced linearized equations
%   [S,K,U,Udot,maskL,dxdxL,dxdu] = fgselim(L,Jx,Ju)
% Based on the Jacobians of the operator Jx and Ju it computes the reduced
% linearized equations of the form:
%   S dxL/dt = K xL + U u + Udot du/dt
% Additional outputs include:
%   - maskL such that xL = x(maskL)
%   - dxdxL and dxdu such that near the linearization point
%     x-x0 = dxdxL*(xL-xL0) + dxdu*(u-u0)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

nS = numel(L.ind.irS); % Number of "static" equations (no xdot terms)

% Determine independent colums of reduced ag constraint jacobian
if L.nC<L.ng % CDE case
  if strcmp(L.code,'rzp')
    % Keep ag(1)=Ip as dynamic state for rzp
    ib = [2,3];
  else
    % Keeping Jyy/Jyg/JCy notation even if state can be Fx
    Jyy = Jx(L.ind.irGS,L.ind.ixGS);
    Jyg = Jx(L.ind.irGS,L.ind.ixg );
    JCy = Jx(L.ind.irC ,L.ind.ixGS);
    JCg = Jx(L.ind.irC ,L.ind.ixg );
    JCgt = JCg - JCy*(Jyy\Jyg);
    [~,ib] = rref(JCgt,1e-3);
  end
else
  ib = 1:L.ng;
end
if (numel(ib) ~= L.nC)
  error('Not enough independent variables were found. Maybe some ag constraints are redundant?');
end

% Separate variables into 1/"static" and 2/"evolutive"
i1 = [L.ind.ixGS,L.ind.ixg(ib)];
i2 = setdiff(1:L.nN,i1);
maskL = ismember(1:L.nN,i2);
n2 = numel(i2);

dx1dx2 = -Jx(1:nS,i1)\Jx(1:nS,i2);
dx1du  = -Jx(1:nS,i1)\Ju(1:nS, :);
dxdxL([i1,i2],:) = [dx1dx2;eye(n2)];
dxdu ([i1,i2],:) = [dx1du ;zeros(n2,size(Ju,2))];

S    =   Jxdot(   1:end,i1)*dx1dx2 + Jxdot(   1:end,i2) ;
K    = -(Jx   (nS+1:end,i1)*dx1dx2 + Jx   (nS+1:end,i2));
U    = -(Jx   (nS+1:end,i1)*dx1du  + Ju   (nS+1:end, :));
Udot = -(Jxdot(   1:end,i1)*dx1du                      ); % No direct udot terms

end
