function LX = fgex(tok,t,L,LX0)
% Get initialization data for fge. 
% function LX = fgex(tok,t,L); % retrieves initial eq from FGS
% function LX = fgex(tok,t,L,LX0); % provide initial eq explicitly
% For information about the content of LX, use help meqt
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin<=3
  fgsxtok = str2func(['fgsx',tok]);
  LX0 = fgsxtok(t(1),L);
end
LX0 = meqxconvert(L,LX0,L); % fill potentially missing fields

if isempty(t)
  t = LX0.t;
end
nt = numel(t);

%% multidomain specials
LX0 = meqxdoublet(L,LX0);

%% Initial condition
% compute initial condition from first time slice of LX0 according to the
% code used (fge or rzp)
if numel(LX0.t)>1, LX0 = meqxk(LX0, 1); end % only slice when needed
LXi = L.codeinit(L,LX0);

%% Time-varying quantities needed
% Need time-varying values for constraints and some other quantities
fields = L.agconc(1:L.nC,3).'; % field used for constraint equations
% other necessary quantities for FGE
fgefields = {'IniD','Ini','signeo', 'Lp', 'Rp'};
fields = [fields,fgefields];

LX = LXi;
LX.t = t;
% get time-varying fields into LX
if nt>1
  for field = fields
    LX0val = LX0.(field{:});
    LXival = LXi.(field{:});
    % keep LXival for first (converged) time slice, 
    % append LX0val in other slices as constraints
    LXval = squeeze(cat(3,LXival,repmat(LX0val,1,1,nt-1)));
    if iscolumn(LXval), LXval = LXval'; end
    LX.(field{:}) = LXval;
  end
end

%% Time-varying feedforward Va trajectory to keep initial equilibrium
% using only first entry of Vadot, Iadot due to linear trajectory
LX.Va = LX.Va(:,1) + LX.Vadot0(:, 1)*(LX.t-LX.t(1));
LX.Ia = LX.Ia(:,1) + LX.Iadot0(:, 1)*(LX.t-LX.t(1));

% Strip some unneeded FBT & LIUQE stuff
fieldsrm = {'Sp','Ep','chi','chie','Fb', 'cycle', 'dz', 'dzg', 'rst', 'chih',...
            'idoublet','werr','uerr','err','ID','Xt'};
LX = rmfield(LX,fieldsrm(isfield(LX,fieldsrm)));

LX = meqx(L,LX); % generic LX handling

% Check for signeo to be nonzero everywhere
assert(~any(LX.signeo(:)<=0), 'Conductivity of plasma has to be strictly positive.')

end
