function LY = fgetkl(L,LX,zLY)
% Step linear system given in state space form
% Return usual LY structure
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

dt = LX.t - zLY.t;

%% Get current state
zxt = L.lin.LX2x(zLY);

%% Update time varying inputs
% Get constraints current time step
Co = LX2Co(L,LX);
zCo = LX2Co(L,zLY);

% Pack time variable inputs u = {Va,Co,Ini,dCodt}; See fgess.m for more details.
ut = zeros(size(L.lin.sysB,2),1);
ut(L.lin.iut) = [LX.Va;Co;LX.IniD;(Co-zCo)/dt];

%% Step system
xt = L.lin.sysA*(zxt-L.lin.x0L) + L.lin.sysB*(ut-L.lin.u0L) + L.lin.x0dotL;
% The last term comes from the non-zero residual at the initial step
% (non-stationary initial conditions)

%% Compute post processing
yt = L.lin.sysC*xt + L.lin.sysD*(ut-L.lin.u0L);
% If linfullstate=true, xt is the new full state, else it is the new perturbed state

%% Add offset
yt = yt + L.lin.y0;

%% Pack outputs into standard LY structure
LY = zLY; % Initialize from previous time slice

% Extracting indices using L.lin.sys.OutputGroup directly is slower
% It is significantly faster to copy the structure first
for ii = fieldnames(L.lin.OutputGroupString)'
  LY.(ii{:}) = reshape(yt(L.lin.OutputGroupString.(ii{:})),meqsize(L,ii{:}));
end

% Set to NaN all the fields which are not computed for the linear post processing
for field = setdiff(fieldnames(zLY),[fieldnames(L.lin.OutputGroupString);'isconverged'])'
  if islogical(LY.(field{:}))
    LY.(field{:})(:) = false;
  else
    LY.(field{:})(:) = NaN;
  end
end

%% Add extra inputs not available in linearization outputs
LY.t = LX.t;
LY.Va = LX.Va;
LY.rBt = LX.rBt;

%% Add flux surface post processing
if L.P.iterq
  % Replace D quantities for single domain for meqpostq
  if L.nD==1
    for ii=fieldnames(LY)'
      str =  ii{:};
      if contains(str,'D')
        LY.(str) = LY.(str(1:end-1));
      end
    end
  end
  [LY.rA,LY.zA,LY.FA,LY.dr2FA,LY.dz2FA,LY.drzFA,LY.rX,LY.zX,LY.FX,~,~,~,...
    LY.rB,LY.zB,LY.FB,LY.lB,LY.lX,LY.Opy,LY.F0,LY.F1] = ...
    meqpdom(LY.Fx,sign(LY.Ip),L.P.isaddl,L);
  [LY.PpQ,LY.TTpQ,LY.PQ,LY.TQ,LY.iTQ,LY.PpQg,LY.TTpQg] = ...
  meqprof(L.fPg,L.fTg,LY.ag,L.pQ(:).^2,LY.F0,LY.F1,LX.rBt,L.bfct,L.bfp,L.idsx,L.smalldia);

  LY.nA=int32(numel(LY.rA)); LY.nB = int32(numel(LY.FB)); LY.nX = int32(numel(LY.FX)); % number of boundaries = number of active domains

  LY.aq = zLY.aq;
  LY.aW = zLY.aW;
  [LY] = meqpostq(L,LY);
  
end

%% TODO make some debugging plot

%% Add sanity check for the state

end
