function [xnlt, LYt, prev_H, solverinfo] = fgetk_implicit(xnlp,xnlpdot,L,LXt,LYp,Prec,prev_H)
% Implicit Euler dt step for NL system
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin<6
  prev_H=[];
end

%% F options
opts = optsF('dopost',true,'doplot',L.P.debugplot>1); % Defaults

%% Initial guess
Iep = [LYp.Ia;LYp.Iu];

% NL residual operator and initial guess
if strcmpi(L.P.algoNL,'picard')
  % initial guess passed via LXt1
  LXt.Ie = Iep; % initial guess: from above
  LXt.Iy = LYp.Iy; % initial guess: previous Iy
  F = @(xnl,opts) fgeFpicard(xnl,L,LXt,LYp,opts);
  xnl0 = xnlp;
else % 'all-nl','all-nl-Fx','Newton-GS'
  % operator
  F = @(xnl,opts) fgeF(xnl,L,LXt,LYp,opts);

  % initial condition
  xnl0 = zeros(L.nN,numel(L.P.initguess));
  for iinit = 1:numel(L.P.initguess)
    switch L.P.initguess{iinit}
      case 'vacuum'
        % initial guess passed via xnl0
        ixe = [L.ind.ixa,L.ind.ixu];
        dt = LXt.t - LYp.t;

        MM = L.Mee; % + L.lin.Xee; % to eventually add plasma part
        Iet = (MM + diag(dt*L.Re))\...
          ([dt*LXt.Va;zeros(L.G.nu,1)] + MM*Iep); % linear step of coil-only circuit equation

        xnlv = xnlp; % update previous solution
        xnlv(ixe) = Iet./L.xscal(ixe); % assign in xnl0
        xnl0(:,iinit) = xnlv;
      case 'extrap'
        dt = LXt.t - LYp.t;
        xnl0(:,iinit) = xnlp + xnlpdot*dt; % linear extrapolation
      case 'previous'
        xnl0(:,iinit) = xnlp; % previous
      otherwise
        error('unknown initial guess option %s',L.P.initguess{iinit})
    end
  end
end

%% Setup solver parameters
P = struct('algoF',L.P.algoF,'mkryl',L.P.mkryl,...
  'relax',L.P.relax,'kmax',L.P.kmax,...
  'debug',L.P.debug,'debugplot',L.P.debugplot, ...
  'debugfail',L.P.debugfail,'debuglinesrch',L.P.debuglinesrch,...
  'tolF',L.P.tolF,'robust_on',1,'algoGMRES',L.P.algoGMRES, ...
  'nrestarts',L.P.nrestarts,'restart_random',L.P.restart_random,...
  'anajac',L.P.anajac,'jacobian_handle',L.P.jacobian_handle,...
  'userowmask',L.P.userowmask,...
  'use_inverse',L.P.use_inverse,'large_system',L.P.large_system,...
  'is_factored',L.P.is_factored,...
  'ldim',L.P.ldim,'group_size',L.P.group_size,...
  'iy',L.ind.ixGS,'prev_H',prev_H,'ratio_gb',L.P.ratio_gb);

% Preconditioner options
if L.P.usepreconditioner
  P.prectype    = L.P.prectype;
  P.precupdate  = L.P.precupdate;
  P.ilu_algo    = L.P.ilu_algo;
  P.ilu_droptol = L.P.ilu_droptol;
  P.Prec        = Prec;
else
  P.prectype    = 'user';
  P.precupdate  = 'once';
  P.Prec        = [];
end

%% Choose the best initial condition
n_init = size(xnl0,2);
if n_init>1
  sqnormFi = zeros(1,n_init);
  for ii=1:n_init
    Fi = F(xnl0(:,ii),optsF());
    sqnormFi(:,ii) = sum(Fi.^2,1);
  end
  % choose solution with the lowest residual
  [~,ibest] = min(sqnormFi);

  xnl0 = xnl0(:,ibest);
  if P.debug>2
    fprintf('\nConsidering %d possible initial guess candidates\n Options: ',n_init)
    for ii=1:n_init, fprintf(  ' %10s',L.P.initguess{ii}); end; fprintf('\n  norm.^2:');
    for ii=1:n_init, fprintf(' %10.4e',sqnormFi(ii)); end
    fprintf('\n ..Chose candidate %d (%s)\n',ibest,L.P.initguess{ibest})
  end
end

%% Solve the equation F(xnl)=0
[xnlt, solverinfo] = solveF(F,xnl0,P);

solution_good_enough = (solverinfo.res < L.P.tolok);

if solverinfo.isconverged || (L.P.keepok && solution_good_enough)
  % Accept this solution, compute post processing
  [~,LYt] = F(xnlt,opts);
  prev_H = solverinfo.prev_H;

  if ~solverinfo.isconverged && L.P.keepok && L.P.debug
    msg = sprintf('Newton solver did not converge, %4.1e<%4.1e so accepting this good enough solution',solverinfo.res,L.P.tolok);
    meqmsge('i',mfilename,L.P.tokamak,LXt.t,solverinfo.niter,...
      L.P.shot,msg,'FGE:keptbest');
  end
else 
  % keep previous solution
  xnlt = xnlp; 
  LYt = LYp;
  % do not update prev_H
  if L.P.keepok && L.P.debug
    msg = sprintf('Newton solver did not converge, %4.1e>%4.1e so keeping old solution',solverinfo.res,L.P.tolok);
    meqmsge('i',mfilename,L.P.tokamak,LXt.t,solverinfo.niter,...
      L.P.shot,msg,'FGE:nosolution');
  end
end

%% other specific FGE outputs directly from LX
% append FGE-specific fields
LYt.Va      = LXt.Va;
LYt.Ini     = LXt.Ini;
LYt.IniD    = LXt.IniD;
LYt.signeo  = LXt.signeo;
LYt.Lp      = LXt.Lp;
LYt.Rp      = LXt.Rp;
LYt.t       = LXt.t;

%% Add convergence info
LYt.res    = solverinfo.res;
LYt.mkryl  = solverinfo.mkryl;
LYt.nfeval = solverinfo.nfeval;
LYt.niter  = solverinfo.niter;
LYt.isconverged = solverinfo.isconverged;

end