% FGE forward solver time evolving default parameters overloading from sources
% P = fgep(P)
% 
% Many parameters are in common with FGSP.
% FGE-specific parameters:
% .stoponalarm Stop simulation if coil alarm is triggered
% .nnoc        Number of subsequent non-converging time steps allowed
% .keepok      If not converged, keep 'good enough' solution for the next iteration
% .tolok       Acceptable tolerance for reusing non-converged solutions (only if keepok=true)
% .initguess   Cell array with initial guesses options to try. Options: 'previous','extrap','vacuum' 
% .lin         Run linearized version 
% .fixplasma   Fix plasma in place during iterations, no Iy update
% .mey         Scales Mey - set to 0 to avoid back-coupling between plasma and vessel currents (for testing).
% .implicit    Solve implicitly or explicit in Ie,Iy
% .ssinit      Initialize to a steady-state condition (possibly modifying coil/vessel currents)
% .wcompa      Weight of regularization term to keep compensation currents small
% .wioha       Weight of regularization term to keep Ohmic current driving currents small
% .compregion  Grid region on which to do compensation ('Iy','Oasx','Oasx-sym', or direct specification)
% .iLpext      Source of Lp (plasma inductance) used for rigid model CDE. true -> provided externally in LX, false -> analytic expression (Iy Myy Iy)/Ip^2 equivalent
% .ctrlfct     Feedback controller function handle (default: [] meaning open-loop simulation)
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function [P] = fgep(P,varargin)

D = fgsp(P,varargin{:}); % start with fgs default parameters

%% Default Parameters
% NL solver parameters 
D.mkryl = 200; % krylov solver space. Can be large since GMRES algoritims terminate automatically. 
D.tolF = 1e-8; % default value of tolF (minimum tolerance max(abs(F(x))))
D.usepreconditioner = true; % if active preconditioner
D.relax = 0; % Apply relaxation on solution

% Broyden Solver parameter are initialized in fgsp

% Input source
D.lin = false; % run linearized version
D.nnoc = 0; % number of consecutive non-converging time steps allowed
D.keepok   = false;
D.tolok    = 1e-2;
D.initguess = {'vacuum'}; % Initial guess options to consider
D.fixplasma = 0; % Fix plasma in place during iterations, no Iy update
D.mey = 1; % Scales Mey - set to 0 to avoid back-coupling between plasma and vessel currents (for testing).
D.implicit = true; % solve implicitly or explicit in Ie,Iy
D.xscal = true; % scale states
D.ssinit     = true; % initialize to a steady-state condition (possibly modifying coil/vessel currents)
% inductive/compensation coils
D.wcompa     = 1.0e-9; % weight of regularization term to keep compensation currents small
D.wioha      = 0.5e-10; % weight of regularization term to keep Ohmic current driving currents small
D.compregion = 'Oasx-sym'; % Grid region on which to do compensation ('Iy','Oasx''Oasx-sym', or direct specification)

D.iLpext = true; % Type of used for rigid model CDE. true -> provided externally in LX, false -> analytic expression (Iy Myy Iy)/Ip^2 equivalent

% Initialization source
D.shot = NaN;
D.ctrlfct = []; % Feedback controller function handle (default: open-loop)

D.stoponalarm = true; % stop simulation if coil alarm is triggered

%% Set defaults
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

%% order
P = orderfields(P);

%% Checks
assert(isnumeric(P.mey) && any(P.mey==[0 1]),'mey0 must be 0 or 1');
assert(P.LYall,'LYall must be true for FGE')

end
