function [res,Jx,Ju,Jxdot] = fgeFlin(L,LX)
% FGEFLIN Returns linearizations of fgeF operator around equilibrium LX
%   [res,Jx,Ju,Jxdot] = fgeFlin(L,LX)
% Done analytically if L.P.anajac or using finite differences otherwise.
%
% The definitions of Jx, Ju and Jxdot are detailed in in the help section
% of FGEL. When computing the jacobians using finite differences, dt must
% be set to 0 by using LYp=LX to compute Jx and Ju, and to compute Jxdot we
% choose LYp=LX except LYp.t=LX.t-1 so that dt=1 and x0=x and we vary xdot
% around 0.
%
% See also FGEF FGEL
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if numel(LX.t)~=1, error('call fgeFlin for only a single time slice'); end

% Non-linear state
x = L.LX2x(LX);

%% Evaluate Jacobian
if L.P.anajac
  opts = optsF('dojacx',true,'dojacu',nargout>2,'dojacxdot',nargout>3);
  [res,~,Jx,Ju,Jxdot] = fgeF(x,L,LX,LX,opts);
else
  opts = optsF;
  res = fgeF(x,L,LX,LX,opts);

  % Perturb w.r.t. state
  dx = 1e-6;
  Jx = jacfd(@fgeF,{x,L,LX,LX,opts},'F0',{res},'szF0',{L.nN},'epsval',dx);

  if nargout>2
    % Perturb w.r.t. inputs
    if L.isEvolutive
      u = [LX.Va(:);LX2Co(L,LX);LX.IniD];
    else
      u = [LX.Ia(:);LX.Iu(:);LX2Co(L,LX)];
    end
    Fb_ext = []; % Leave empty to be recomputed
    du = 1e-6;
    Ju = jacfd(@fgeF,{x,L,LX,LX,opts,Fb_ext,u},'F0',{res},'szF0',{L.nN},'iargin',7,'epsval',du);
  end

  if nargout>3
    % Perturb w.r.t. previous state
    if L.isEvolutive
      % fgeF with xdot as argument requires dt>0
      LYp = LX;
      LYp.t = LX.t - 1;
      xdot = zeros(L.nN,1);
      dxdot = 1e-6;
      Jxdot = jacfd(@fgeF,{x,L,LX,LYp,opts,Fb_ext,u,xdot},'F0',{res},'szF0',{L.nN},'iargin',8,'epsval',dxdot);
      % Remove derivatives of static residuals
      Jxdot = Jxdot(L.ind.irD,:);
    else
      Jxdot = zeros(0,L.nN);
    end
  end
end
end
