%FBTXANA  ANA FBT equilibrium specification
% LX = FBXPANA(t,L) returns a structure LX with cost function/
% constraint data for the ANA tokamak.
%
% Several shots are pre-defined:
%
%     0: no constraints and zero plasma current
%     1: limiter
%     2: single null
%     3: single null with secondary x point
%     4: double null
%     5: limited kappa<1
%     6: negative triangularity limited
%    11: Elongated
%    21: Circular with X point outside limiter
%    81: Droplets
%    82: Doublet
%    83: Diverted droplet with second droplet in private flux region
%    84: Diverted droplet with second droplet in public flux region
%    85: Two diverted droplets
%    91: Vacuum case with quadrupole field
%
% Time-dependent shots:
% 
%   101: Current ramp up to limiter equilibrium 1 and back
%   201: Same as 101 with time derivative constraints
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function LX = fbtxana(t,L)

% Recursive call for time dependent shots
if L.P.shot>100 % time-dependent shots
  LX = fbtxana_timedependent(L);
else
  LX = fbtxana_single(L);
end

if ~isempty(t)
  LX = fbtxinterp(LX,t,L.P.interpx);
end

end

function LX = fbtxana_single(L)
% Single equilibrium case

P  = L.P;
LX = struct();
% flux points
npts = 8;
thp  = linspace(0,2*pi,npts+1)'; thp=thp(1:end-1);
a0 = 0.40;
z0 = 0;

% defaults
LX.t        = 0;
LX.rBt      = 1*P.r0;
LX.Ip       = 200e3;
LX.isaddl   = 1;
LX.niter    = L.P.niter; 
LX.idoublet = false;
LX.shot     = P.shot;

% shot dependent overrides
switch floor(LX.shot/10)
  case 8
    % doublets
    LX.idoublet = true;
    LX.IpD = LX.Ip*[0.5;0.5;0];
    LX.bpD = [0.25;0.25;0];
    LX.qA  = [1;1];
  otherwise
    % singlets
    LX.bp  = 1;
    LX.qA  = 1;
end

%% Individual equilibrium definitions
switch LX.shot
  case 0 % no plasma, no constraints
    LX = fbtgp(LX,[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[],[]);
    LX.Ip = 0;
  case 1 % limited, circular
    %         bouD(n,r0  ,z0  ,a0 ,k1 ,k2 ,d1 ,d2 ,l1 ,l2  ,p1,p2)
    [ra,za] = bouD(9,1.05,0.  ,a0 ,1.0,1.0,0.0,0.0,0.0,0.0 ,0 ,0 );
    %     fbtgp(X,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1 ,0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Exact boundary point
    LX.gpfe(1) = 0;
  case 2 % diverted
    %         bouD(n,r0  ,z0  ,a0 ,k1 ,k2 ,d1 ,d2 ,l1 ,l2  ,p1,p2)
    [ra,za] = bouD(9,1.05,0.05,0.3,1.3,1.5,0.1,0.6,0.1,-0.3,0 ,0 );
    %    fbtgp(LX,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Exact boundary point
    LX.gpfe(1) = 0;
    % Specify an X-point
    rX = 0.85; zX = -0.4;
    %    fbtgp(LX,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,rX,zX,1,0 ,1 ,0 ,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[], []);
  case 3 % diverted with secondary x-point
    %         bouD(n,r0  ,z0   ,a0 ,k1 ,k2 ,d1 ,d2 ,l1 ,l2  ,p1,p2)
    [ra,za] = bouD(8,1.05,-0.03,0.28,1.2,1.3,0.3,0.8,0.1,0.1 ,0 ,0 );
    %    fbtgp(P ,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify a primary X-point
    rX = 0.775; zX = -0.39;
    %    fbtgp(P ,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,rX,zX,1, 0, 1, 0,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
    % Specify a secondary X-point
    rX = 0.775; zX = 0.39;
    %    fbtgp(P ,r ,z ,b,fa,fb,fe ,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,rX,zX,0, 0, 1,Inf,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
  case 4 % double null diverted
    r0 = P.r0 + 0.02;
    a0 = 0.4;
    % X points
    rX = r0 - [1; 1]*a0*0.5;
    zX = z0 + [1;-1]*a0*1.0;
    % boundary and strike points
    rr = [1; 1]*(r0+[0.25 0.0 -0.23 -0.19 -0.28]);
    zz = [1;-1]*[0.1 0.32 0.2 0.45 0.40];
    r = [rX;rr(:)]; z = [zX;zz(:)];
    
    LX = fbtgp(LX,r,z,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify an X-point
    LX.gpfe(1:2)   = 0; % Make X-points exact constraints
    LX.gpbr([1,2]) = 0; % Br == 0
    LX.gpbz([1,2]) = 0; % Bz == 0
    LX.gpbe([1,2]) = 0; % B field constraints are exact
    % Remove strike points from initial guess
    LX.gpb(end-3:end) = 0;
  case 5
    % squashed (vertically stable)
    r0 = P.r0 + 0.05;
    ra = r0 + a0*cos(thp);
    za = z0 + 0.9*a0*sin(thp);
    %   fbtgp(P,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Exact boundary point
    LX.gpfe(2) = 0;
  case 6 % NT limited with secondary x points
    %         bouD(n,r0  ,z0 ,a0  ,k1  ,k2  ,d1  ,d2  ,l1 ,l2 ,p1,p2)
    [ra,za] = bouD(8,0.88,0.0,0.33,1.15,1.15,-0.6,-0.6,0.2,0.2,0 ,0 );
    %    fbtgp(P ,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Secondary X points
    rX = 1.22*[ 1; 1];
    zX = 0.40*[ 1;-1];
    %    fbtgp(P ,r ,z ,b,fa,fb,fe ,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,rX,zX,0, 0, 1,Inf,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
  case 81 % two droplets
    % Upper droplet
    z0 = 0.55; k = 1; a0 = 0.35;
    [ra,za] = bouD(11,P.r0,+z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Lower droplet
    [ra,za] = bouD(11,P.r0,-z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
  case 82 % doublet!
    % Upper droplet
    r0 = P.r0; z0 = 0.45; a0 = 0.25; k = 1.2; npt = 11;
    [ra,za] = bouD(npt,r0,+z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Lower droplet
    [ra,za] = bouD(npt,r0,-z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify an X-point
    rX = r0; zX = 0;
    LX = fbtgp(LX,rX,zX,0,[],[],[],0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
  case 83 % diverted droplet with second droplet in private flux region
    % Upper limited droplet
    r0 = P.r0; z0 = +0.65; a0 = 0.21; k = 1.2; npt = 11;
    [ra,za] = bouD(npt,r0, z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1,10,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Lower diverted droplet
    z0 = -0.35;
    [ra,za] = bouD(npt,r0, z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify an X-point
    rX = r0; zX = 0.15;
    LX = fbtgp(LX,rX,zX,0, 0, 0, 0,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
  case 84 % diverted droplet with second droplet in public flux region
    % Upper diverted droplet
    r0 = P.r0; z0 = +0.5; a0 = 0.25; k = 1.4; npt = 11; l = -0.4;
    [ra,za] = bouD(npt,r0, z0,a0,k,k,0,0,l,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify an X-point
    rX = r0; zX = z0+a0*k;
    LX = fbtgp(LX,rX,zX,0, 0, 1, 0,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
    % Lower limited droplet
    r0 = P.r0; z0 = -0.55; a0 = 0.35; k = 1.0; npt = 11; l = -0.2;
    [ra,za] = bouD(npt,r0,z0,a0,k,k,0,0,l,l,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
  case 85 % two diverted droplets
    r0 = P.r0; z0 = 0.5; a0 = 0.25; k = 1.4; npt = 11; l = -0.3;
    [ra1,za1] = bouD(npt,r0, z0,a0,k,k,0,0,l,0,0,0);
    [ra2,za2] = bouD(npt,r0,-z0,a0,k,k,0,0,0,l,0,0);
    ra = [ra1;ra2]; za = [za1;za2];
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % X-points
    rX = r0*[1;1]; zX = (z0+a0*k)*[1;-1];
    LX = fbtgp(LX,rX,zX,0, 0, 1, 0,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
  case 88 % doublet with mantle current
    % Upper droplet
    r0 = P.r0; z0 = 0.45; a0 = 0.25; k = 1.2; npt = 11;
    [ra,za] = bouD(npt,r0,+z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Lower droplet
    [ra,za] = bouD(npt,r0,-z0,a0,k,k,0,0,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    % Specify an X-point
    rX = r0; zX = 0;
    LX = fbtgp(LX,rX,zX,0,0,0,0,0 ,0 ,[],0 ,[],[],[],[],[] ,[] ,[] ,[]);
    LX.IpD = LX.Ip*[0.49;0.49;0.02];
    LX.gpz = LX.gpz + 0.03; % shift everything up to force limiter point
  case 11 % high elongation with triangularity
    z0 = 0.002; k = 2.2; d = 0.5; a = 0.35;
    [ra,za] = bouD(11,P.r0+0.1, z0,a,k,k,d,d,0,0,0,0);
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    LX.Ip = 500e3;
  case 21 % circular with X-point outside limiter
    r0 = P.r0 + 0.05;
    ra = r0 + a0*cos(thp);
    za = z0 + a0*sin(thp);
    %    fbtgp(LX,r ,z ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,ra,za,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
    rx = r0 + .9*a0;
    zx = z0 + .9*a0;
    LX = fbtgp(LX,rx,zx,0, 0, 0,Inf,0,0,NaN,0,[],[],[],[],[] ,[] ,[] ,[]);
  case 91 % vacuum case with quadrupole null
    % fbtgp(X,    r     ,    z ,b ,fa,fb,fe,br,bz,ba,be,cr ,  cz,ca,ce,vrr,vrz,vzz,ve)
    LX = fbtgp(LX,L.P.r0,    0 ,0 ,[],[],[],0 ,0 ,[], 0,1,[] ,[] ,0,[],[],[],[]);
    LX.Ip = 0;
  otherwise
    error('Shot %d not available for anamak',LX.shot);
end

na = L.G.na; % number of circuits
nu = L.G.nu; % number of passive currents
no = L.G.no; % number of circuit equality constraints

% Cost function weights
fac = LX.Ip/L.Ip0;
if fac == 0; fac=1; end % plasmaless case
LX.gpfd = 1e-2*L.Fx0*fac;
LX.gpbd = LX.gpfd;
LX.gpcd = LX.gpfd;
LX.gpid = 2e5*fac;
LX.gpdd = Inf;
LX.gpud = 1.3e4*fac;
LX.gpad = Inf;
LX.gpod = LX.gpid;

% Targets
LX.gpia = zeros(na,1);      % Circuit current target
LX.gpua = zeros(nu,1);      % Passive current target
LX.gpaa = zeros(na,1);      % Circuit voltage target
LX.gpoa = zeros(no,1);      % Circuit constraint target

% Dipoles
LX.gpdw = ones (na,1);      % Equal weights for all current dipoles

end

function LX = fbtxana_timedependent(L,shot)
if nargin==1
  shot = L.P.shot;
end
% time-dependent anamak shots based on combining static equilbria
switch shot
  case 101
    t = [0 0.01 0.03 0.05 0.1 0.9 0.95 0.97 0.98 1];
    eq = 1*ones(size(t)); % equilibrium number per time
    for kt = 1:numel(t)
      L.P.shot = eq(kt);
      LX(kt) = fbtxana_single(L);
      assert(numel(LX(kt).t)==1,'expected one time slice')
      LX(kt).t = t(kt);
    end
    LX = meqlpack(LX);
    LX.Ip = [50 80  120 170 200]*1e3; LX.Ip = [LX.Ip ,fliplr(LX.Ip)];
    LX.qA = [2  1.5 1.2 1.1   1]    ; LX.qA = [LX.qA ,fliplr(LX.qA)];

    % Scale errors for all eq according to IpD
    list = strcat('gp',{'f','b','c','v','i','o','u','a'},'d');
    for field = list
      LX.(field{1}) = LX.(field{1}) .* LX.Ip./2e5; % Original eq has Ip=200kA
    end

  case 201
    % copy 101
    LX = fbtxana_timedependent(L,101);
    % add time derivative constraints
    nt = numel(LX.t);
    r  = repmat(L.P.r0,1,nt);
    z  = zeros(1,nt);
    fe = zeros(1,nt);
    %    fbtgp( X,r,z,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve,timeder)
    LX = fbtgp(LX,r,z,0, 1, 0,fe,[],[],[],[],[],[],[],[], [], [], [],[],      1); % Set dFdt=Vloop(r=r0,z=0) = 1
    LX.g1fe([1,end]) = Inf; % Deselect time derivative constraints for first and last points
    LX.g1fd(:) = 1;
    fe = ones(1,nt);
    %    fbtgp( X,r,z,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve,timeder)
    LX = fbtgp(LX,r,z,0, 0, 0,fe,[],[],[],[],[],[],[],[], [], [], [],[],      2); % Minimize d2Fdt2=dVloopdt(r=r0,z=0) = 0
    LX.g2fe([1,end]) = Inf; % Deselect time derivative constraints for first and last points
    LX.g2fd(:) = 1;
  otherwise
    error('time-dependent %d shot not defined',L.P.shot);
end

% Set LX.shot
LX.shot(:) = shot;

end
