function LX = fbtx(L,LX)
% function LX = fbtx(L,LX)
% Defaults and general things for fbtx
%
% LX fields in FBT:
%
% Fields that should come from fbtxtok:
%  .t        Time grid
%  .IpD      Plasma current per domain
%  .bpD      betap per domain
%  .qA       Axis q per domain
%  .rBt      R*Bt
%  .gp*      Various equilibrium constraint/cost function parameters documented in FBTGP and FBTHELP
%  .idoublet This time slice is a doublet
%  .bfp      (optional) time-varying basis function parameters when using bffbt
%
% Fields that should come from fbtp or can be passed as arguments (see help FBTP):
%  .niter
%  .tol
%  .capaj
%
% Fields from meqp:
%  .isaddl
%  .idoublet
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

assert(nargin==2,'fbtx needs two input arguments, L and LX');
nt = numel(LX.t);
if nt == 0, error('fbtx needs an LX structure with at least 1 time slice'); end

%% defaults
pfields = {'idoublet','isaddl','tol','capaj','niter'};
for field=pfields
  myfield = field{1};
  if isfield(LX,myfield)
    % if LX already has them
    assert(size(LX.(myfield),2)==nt,...
      'LX.%s has incorrect number of time slices, expect %d found %d',myfield,numel(LX.t),size(LX.(myfield),2))
    continue % just check size
  else
    if isfield(L.P,myfield)
      % copy value from L.P.myfield
      pvalue = L.P.(myfield);
      if size(pvalue,2) == 1 
        LX.(myfield) = repmat(pvalue,1,nt); % repeat single value across all times
      elseif size(pvalue,2) == nt
        LX.(myfield) = pvalue; % copy array of time-dependent values
      elseif size(pvalue,2) == numel(L.P.t)
        LX.(myfield) = pvalue(:,iround(L.P.t,LX.t)); % if still on P.t time base from fbtptcv
      else, error('invalid size for L.P.%s. Expected single or %d time slices, found %d',...
          myfield,nt,size(pvalue,2))
      end
    else
      error('LX.%s is not provided nor can a default be found in L.P',myfield)
    end
  end
end

%% bffbt special
if ~isfield(LX,'bfp')
  isbffbt = strcmp(func2str(L.P.bfct),'bffbt'); % Only support specifying bffbt directly
  if isbffbt
    LX.bfp = L.bfp;
  end
end

%% multidomain special
LX = meqxdoublet(L,LX);

%% Value checks
if L.nD == 1, IpD = LX.Ip;
else,         IpD = LX.IpD(1:size(LX.qA,1),:);
end
% Check sign congruence of qA,rBt,Ip
assert(all(~any(LX.qA.*LX.rBt.*IpD < 0) ),'signs of qA,rBt,Ip are not congruent, product of signs must be positive')

%% Fields for profile constraints
fields = L.agconc(:,3).';
for field = fields
  if strcmp('fbtlegacy',field{:}), continue; end
  if ~isfield(LX,field{:}) || any(isnan(LX.(field{:})(:))) % populate missing or NaN fields
    switch field{:}
      case 'Wk'
        if isfield(LX,'bp')
          LX.Wk  = LX.bp .*(1.5e-7*pi*L.P.r0*LX.Ip .^2);
        else
          LX.Wk = NaN(1,nt);
        end
      case 'WkD'
        if isfield(LX,'bpD')
          LX.WkD = LX.bpD.*(1.5e-7*pi*L.P.r0*LX.IpD.^2);
        else
          LX.WkD = NaN(L.nD,nt);
        end
      case 'ag'
        LX.ag  = zeros(L.ng,nt);
    end
  end
  if ~isfield(LX,field{:})
    error('fbtx:agcon','Field %s is missing from the LX structure but is needed in order to constrain the profiles',field{:});
  end
end
if ~any(ismember({'bp','bpD'},fields))
  if isfield(LX,'bp' ), LX.bp (:) = NaN;end
  if isfield(LX,'bpD'), LX.bpD(:) = NaN;end
end

%% CDE
LX.IniD = zeros(L.nD,nt);
LX.Ini  = zeros(   1,nt);

%% Merge constraints that can be merged
if L.P.mergex
  for kt=1:nt
    LXt = meqxk(LX,kt);
    iok = ~any(isnan([LXt.gpr,LXt.gpz]),2);
    % grouping Br=0 and Bz=0 constraints for X-points.
    xx = find(iok & ((LXt.gpbr==0 & LXt.gpbd*LXt.gpbe==0) | (LXt.gpbz==0 & LXt.gpbd*LXt.gpbe==0)));
    [C,~,iC] = unique([LXt.gpr(xx),LXt.gpz(xx)],'rows');
    for ii = 1:size(C,1)% for each unique one
      ioccurrence = xx(iC==ii);
      if numel(ioccurrence)>1 && all(any([LXt.gpbr(ioccurrence),LXt.gpbz(ioccurrence)]==0)) % more than one occurrence and both Br=0 and Bz=0
        ikeep = ioccurrence(1    ); % keep first
        ielim = ioccurrence(2:end); % eliminate others
        LX.gpbr(ikeep,kt) = 0; LX.gpbz(ikeep,kt) = 0;   LX.gpbe(ikeep,kt)=0;
        LX.gpbr(ielim,kt)=NaN; LX.gpbz(ielim,kt) = NaN; LX.gpbe(ielim,kt)=Inf;
      end
    end
  end
end

%% Group gp* by same gpr,gpz points
list = strcat('gp',{'r','z','b','fa','fb','fe','br','bz','ba','be','cr','cz','ca','ce','vrr','vrz','vzz','ve'});
for kt = 1:nt
  [~,ia,ic] = unique([LX.gpr(:,kt),LX.gpz(:,kt)],'rows'); % ia(ic) is the index of the first occurence of each row
  [~,sorti] = sort(ia(ic));                               % This reorders rows to group them by value while keeping the order stable
  for field = list
    nrow = size(LX.(field{1}),1);
    assert(nrow==numel(sorti),'field %s has unexpected number of rows, expected %d, found %d',field{1},numel(sorti),nrow);
    LX.(field{1})(:,kt) = LX.(field{1})(sorti,kt);
  end
end

%% Defaults for control point terms
% Cost function weights

fac = sum(IpD,1)/L.Ip0;
if fac == 0; fac = 1; end % plasmaless case
if ~isfield(LX,'gpfd'), LX.gpfd = 1e-2*L.Fx0*fac; end
if ~isfield(LX,'gpbd'), LX.gpbd = LX.gpfd; end
if ~isfield(LX,'gpcd'), LX.gpcd = LX.gpfd; end

%% Defaults for current-related terms
if ~isfield(LX,'gpue')
  if L.P.circuit % if solving circuit equations, do not constrain first
    LX.gpue = [Inf(L.G.nu,1),ones(L.G.nu,nt-1)];
  else
    LX.gpue = zeros(L.G.nu,nt); % constrain all
  end
end

% Global errors for coil/passive currents, coil currents dipoles/combinations, coil voltages
if ~isfield(LX,'gpid')
  if isfield(L.P,'dissi') % To be removed
    LX.gpid = LX.gpfd./(L.G.na*2*pi*sqrt(2*L.P.dissi ));
  else
    Ia0max = max(abs(L.Ia0));
    if isempty(Ia0max), Ia0max = 1; end
    LX.gpid = repmat(Ia0max,1,nt);
  end
end
if ~isfield(LX,'gpdd')
  if isfield(L.P,'dipol') % To be removed
    LX.gpdd = LX.gpfd./(       2*pi*sqrt(  L.P.dipol ));
  else
    LX.gpdd = LX.gpid;
  end
end
if ~isfield(LX,'gpod'), LX.gpod = LX.gpid; end
if ~isfield(LX,'gpud')
  if isfield(L.P,'dipas') % To be removed
    LX.gpud = LX.gpfd./(L.G.nu*2*pi*sqrt(  L.P.dipas ));
  else
    Iu0max = max(abs(L.Iu0));
    if isempty(Iu0max), Iu0max = 1; end
    LX.gpud = repmat(Iu0max,1,nt);
  end
end
if ~isfield(LX,'gpad')
  if isfield(L.P,'divolt') % To be removed
    LX.gpad = LX.gpfd./(L.G.na*2*pi*sqrt(  L.P.divolt));
  end
end

if L.P.circuit
  % Since the first slice is run with d/dt=0, Iu=0 is directly imposed by the
  % plasma model, gpue should not interfere.
  %   Check gpue=Inf/NaN
  assert(~any(isfinite(LX.gpue(:,1))),'gpue should be Inf for the first equilibrium with P.circuit=true');
else
  % Check gpue=0 for all equilibria
  assert(~any(LX.gpue(:)),'gpue should be 0 for all equilibria with P.circuit=false');
end

% Check LX.bfp values
if isfield(LX,'bfp')
  bfpck(L.bfct,LX.bfp,L.nD);
end


%% Fill in any missing defaults for current/voltage related equations
% maximum time derivative order to consider
if any(startsWith(fieldnames(LX),'g2')), max_derorder = 2;
elseif any(startsWith(fieldnames(LX),'g1')), max_derorder = 1;
else, max_derorder = 0;
end
assert(max_derorder==0 || L.P.circuit,'must solve circuit equation to enable time derivative terms in LX')

for derorder = 0:max_derorder
  LX = gp_check_and_fill_defaults(L,LX,derorder);
end

%% meqx for general additions and further defaults
LX = meqx(L,LX);

%% Verbosity
if L.P.debug>1
  for kt=1:numel(LX.t)
    fbtxdisp(L,meqxk(LX,kt));
  end
end

end

function LX = gp_check_and_fill_defaults(L,LX,derorder)
% check field default sizes for gp* fields
% and fill the missing ones with values that will
% cause them to be ignored.

% field name prefix per derivative order
switch derorder
  case 0, gp = 'gp';
  case 1, gp = 'g1';
  case 2, gp = 'g2';
end

gpr = [gp,'r']; gpz = [gp,'z']; gpb = [gp,'b'];

nt = numel(LX.t);
if ~isfield(LX,gpr)
  % add empty defaults
  LX.(gpr) = zeros(0,nt);
  LX.(gpz) = zeros(0,nt);
  LX.(gpb) = zeros(0,nt);
end
npts = size(LX.(gpr),1);

% check gpr,gpz first
assert(size(LX.(gpr),2) == nt,'%s has incorrect number of time slices',gpr)
assert(size(LX.(gpz),2) == nt,'%s has incorrect number of time slices',gpz)
assert(size(LX.(gpz),1) == npts,'%s, %s have different number of points',gpr,gpz)
% check gpb size and value
assert(all(size(LX.(gpb)) == [npts,nt]),'%s has incorrect size',gpb)
assert(all(LX.(gpb)(:)==0 | LX.(gpb)(:)==1),'%s must be 0 or 1',gpb)

% Check all other entries
for eqcat={'i','d','o','u','a','f','b','c','v'} % equation category
  % entries to check per equation, and number of expected rows
  switch eqcat{:}
    case 'i', n = L.G.na; entries = {'a','e','d'};
    case 'd', n = L.G.nd; entries = {'a','e','d'};
    case 'o', n = L.G.no; entries = {'a','e','d'};
    case 'u', n = L.G.nu; entries = {'a','e','d'};
    case 'a', n = L.G.na; entries = {'a','e','d'};
    case 'f', n = npts; entries = {'a','e','d'};
    case 'b', n = npts; entries = {'a','e','d','r','z'};
    case 'c', n = npts; entries = {'a','e','d','r','z'};
    case 'v', n = npts; entries = {'e','d','rr','zz','rz'};
  end
  for entry = entries
    LX = check_and_fill_defaults_per_field(LX,gp,eqcat{:},entry{:},n,nt);
  end
end

% gpdw special case
field = [gp,'dw'];
if ~isfield(LX,field)
  LX.(field) = ones(L.G.na,nt);
else
  val = LX.(field);
  % check size
  assert(all(size(val) == [L.G.na,nt]),'incorrect size for %s',field)
  % check value
  assert(all(isfinite(val(:))),'entries of %s must be finite',field)
end

% gpfb special case
field = [gp,'fb'];
if ~isfield(LX,field)
  LX.(field) = zeros(L.G.na,nt);
else  
  val = LX.(field);
  % check size
  assert(all(size(val) == [npts,nt]),'incorrect size for %s',field)
  % check value
  assert(all(any(val(:)==[0,1],2) | isnan(val(:))),'entries of %s must be 0,1 or NaN',field)
end

end

function LX = check_and_fill_defaults_per_field(LX,gp,eqcat,entry,n,nt)
field = [gp,eqcat,entry];
if ~isfield(LX,field) % field does not exist, set default
  % target value
  switch entry
    case 'd', val = Inf(1,nt);
    case 'e', val = ones(n,nt);
    case 'a', val = zeros(n,nt);
    otherwise
      val = NaN(n,nt);
  end
  LX.(field) = val;
else % field exists, check size
  % expected size
  if isequal(entry,'d'), m = 1;
  else, m = n;
  end
  val = LX.(field); szval = size(val);

  % validate type and permitted value
  switch entry
    case 'e' % weights must be positive
      assert(all(isnumeric(val(:)) & ~isnan(val(:)) & val(:)>=0),...
        '%s must be numeric, not NaN, and nonnegative. Use Inf to signal inactive equation',field);
    case 'd'
      assert(all(isnumeric(val(:)) & ~isnan(val(:)) & val(:)>0),... % strictly larger than 0
        '%s must be numeric, not NaN, and positive. Use Inf to signal inactive equation',field);
    otherwise
      assert(all(isfinite(val(:)) | isnan(val(:))),'%s must be finite or NaN (not Inf).',field);
  end
 
  % check size
  assert(all(szval==[m,nt]),...
    '%s size [%d,%d] does not match expected size: [%d,%d]',...
    field,szval(1),szval(2),m,nt)
end
end
