function [d1dt,d2dt] = fbttimeder(t,type)
% FBTTIMEDER Stencils for time derivative computations in FBT
%
%   [D1DT,D2DT] = FBTTIMEDER(T,TYPE)
%
% For a time sequence T, compute the stencils D1DT and D2DT such that for
% any vector F corresponding to the values of function at times T,
% D1DT*F and D2DT*F correspond to estimates of the first and second
% derivatives of that function at times T. The sequence T need not be
% regularly spaced.
% The optional argument TYPE specifies the type of estimate for the first
% derivative, options are 'backward' (default) and 'centered'.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

t = t(:);
nt = numel(t);

% Add points before first and after last
d1dt = zeros(nt+2,nt+2);
d2dt = zeros(nt+2,nt+2);
dt = vertcat(Inf,diff(t),Inf);

assert(all(diff(t)>0),'t must be monotonously increasing')

for it = 2:nt+1
  dtp = dt(it-1); % dt to previous time point
  dtn = dt(it);   % dt to next time point
  switch type
    case 'centered'
      % 1st order time-derivative zeroing 2nd order terms (centered FD for regular time grid)
      d1dt(it,it-1) = -dtn/dtp/(dtn+dtp);
      d1dt(it,it  ) = 1/dtp - 1/dtn;
      d1dt(it,it+1) =  dtp/dtn/(dtn+dtp);
    case 'backward'
      % 1st order time-derivative
      d1dt(it,it-1) = -1/dtp;
      d1dt(it,it  ) =  1/dtp;
  end
  % 2nd order time-derivative
  d2dt(it,it-1) =  2/(dtn+dtp)/dtp;
  d2dt(it,it  ) = -2/dtn/dtp;
  d2dt(it,it+1) =  2/(dtn+dtp)/dtn;
end

% Remove extra points
d1dt = d1dt(2:end-1,2:end-1);
d2dt = d2dt(2:end-1,2:end-1);

end