%FBTP  FBTE default configuration parameters
% P = FBTP(P,'PAR',VAL,...) fills the structure P with default
% configuration parameters, optionally replacing or adding parameters with
% specified values.
%
% FBT-specific parameters:
% .fbtagcon    Internal profile constraints (see meqagcon for details)
% .limm        Margin factor for coil current limits, values lower than 1 are more conservative. Can be a scalar or a column vector with as many elements as limits.
% .mergex      If true, merge redundant X-point constraints in LX
% .circuit     If true, solve time-dependent problem with circuit equations (GSpulse algorithm)
% .itert       If not 0, re-use previous Iy from previous time as starting guess
% .iterfrz     Number of iterations before which plasma domain is frozen to ease convergence
% .interpx     Interpolation method for LX time base. Options: 'none', 'nearest','linear','previous','next'
% .MHyinterp   If true, use qintmex interpolation of Mxy from meqFxjac for computing control point sensitivity MHy
% .qpsolver    Solver for contrained QP problem. 'quadprog' (default if available), or 'ipm-optimized' (fallback) or 'ipm' (see HELP IPM).
% .useSQP      If true, use sequential quadratic programming
% .wsSQP       Number of picard iterations to perform to warm-start the SQP iterations
% .initineq    If true, inequality constraints will be included to compute the initial guess
%
% .algoNL      Plasma convergence method
% .xscal       If true, scale internal plasma state as in FGS
% .tolstep     Tolerance for step size
% .timeder     Type of stencil for time derivatives, 'centered' or 'backward'
% .jacobian_handle    Use function handle form of jacobian
% .jacobian_sparse    Use sparse-matrix form of jacobian (algoNL=Newton-GS)
%
% SQP Solver parameters
% .userowmask  Reduce Newton problem size when possible
% .algoGMRES   Algorithm variant for Block-GMRES method (direct_inversion,sim,giv,qr)
% .mkryl       Maximum krylov subspace dimension for Block-GMRES
% .epsilon_res Residual tolerance for Block-GMRES
% .usepreconditioner  If true, use Picard approx. preconditioner for Block-GMRES
%
% Quantities which will end up in LX
% .tol         Tolerance for residual
% .capaj       High capaj means stronger peaking of initial plasma current density guess
% .niter       Maximum number of iterations
%
% if bffbt basis function is used, additional parameters exist which are documented in BFPFBT
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function P = fbtp(P,varargin)
 
 %% Default parameters
 D = meqp(); 
 D.fbtagcon = {'Ip','Wk','qA'}; % default constraints (see meqagcon for details)
 D.mergex   = true;
 D.selu     = 'n'; % no vessel currents in fbt 
 D.limm     = 1;   % Use default coil current limits
 D.circuit  = false;
 D.itert    = 0; 
 D.iterfrz  = 50;
 D.voltlim  = false; % impose voltage limits
 D.interpx  = 'nearest'; % nearest neighbour interpolation allowed by default
 % quantities which will end up in LX
 D.tol      = 1e-6;
 D.capaj    = 0.5;
 D.niter    = 100;
 D.MHyinterp= true;

 D.qpsolver = 'ipm-optimized';

 D.useSQP   = false;
 D.wsSQP    = 2;
 D.initineq = false;
 D.algoNL   = 'all-nl';
 D.xscal    = true;
 D.tolstep  = 1e-8;
 D.timeder  = 'backward';
 D.jacobian_handle = false;
 D.jacobian_sparse = false;

 D.userowmask = true;
 D.algoGMRES = 'direct_inversion';
 D.mkryl     = 200;
 D.epsilon_res = 1e-12;
 D.usepreconditioner = true;

 %% Overwrites and set defaults
 for k = 1:2:length(varargin), P.(varargin{k}) = varargin{k+1}; end
 for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
 end
 
 %% Base function parameters
 if isequal(func2str(P.bfct),'bffbt')
   P.bfp = bfpfbt(P);
 end

 %% Check configuration parameters
 msgid = 'FBT:ParameterNotSupported';
 msge = 'FBT Parameter error: %s %g';
 assert(isscalar(P.limm) || (iscolumn(P.limm) && numel(P.limm) == numel(P.limu)),msgid,msge,...
   'limm should be a scalar or a column vector with as many elements as limu');

 if strcmp(P.qpsolver,'quadprog') && isempty(which('quadprog'))
   error('FBTP:NoQuadProg','QP solver ''quadprog'' selected but not found (likely missing optimization toolbox)')
 end

 % Check for deprecated FBT weight parameters
 deprecated  = {'dissi','dipol','strki','dipas','divolt'};
 replacement = {'gpid' ,'gpdd' ,'gpdd' ,'gpud' ,'gpad'  };
 check = isfield(P,deprecated);
 for ii = find(check)
   warning('FBT:deprecatedP','Parameter %s has been deprecated, please assign LX.%s directly',deprecated{ii},replacement{ii});
 end
end
