function [Rax,Rau,Rab,Ra0,...
          Rex,Reu,Reb,Re0,...
          Rix,Riu,Rib,Ri0,...
          ID,w,bb] = fbtopt(L,GP,idoublet)
% FBTOPT Compute FBT optimization problem for single time slice
%
% The FBT optimization problem aims at finding [x,u,Fb] that minimizes the
% cost function
%   W(x,u,Fb) = 1/2*|Rax*x + Rau*u + Rab*Fb - Ra0|^2
% under the exact constraints
%   Rex*x + Reu*u + Reb*Fb - Re0 = 0
% and linear inequality constraints
%   Rix*x + Riu*u + Rib*Fb - Ri0 <= 0
% x is the plasma state variable, u is a set of external inputs and Fb is
% a fitting parameter. Additionally x and u must verify the FGS or FGE set
% of equations F(x,u)=0.
%
% In the time-independent case, the external inputs are [Ia;Iu;Co] but Iu
% and Co are exactly prescribed so the external inputs are limited to u=Ia.
% The plasma model equations correspond to the fgeF operator in the
% non-evolutive case.
%
% In the time-dependent case, the external inputs are [Va;IniD;Co] but IniD
% and Co are exactly prescribed so the external inputs are limited to u=Va.
% The plasma model equations correspond to the fgeF operator with time
% evolution and the optimization is done over all (x,u,Fb) values for a
% given interval. For the first time slice of the interval we assume a
% steady-state condition such that Va=Ra*Ia and Iu=0 (no CDE considered
% yet) which is appropriate for breakdown scenarios (after the
% pre-magnetization phase).
%
% Additional outputs include:
%   ID  Equation id for each cost function term
%   w   Weight for each cost function term
%   bb  bounding box of plasma current initial guess
%
% SEE ALSO FBTHELP, FBTLSQINIT
% 
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Prerequisites
% Impose voltage constraints
na = L.G.na; % number of active coil currents
nu = L.G.nu; % number of passive coil currents
no = L.G.no; % number of coil current constraints
% use passive current evolution

%% Geometry
rH  = GP.gpr;
zH  = GP.gpz;
bH  = GP.gpb;
kH  = ~isnan(rH) & ~isnan(zH) & ~isnan(bH);
rH  = rH(kH);
zH  = zH(kH);
bH  = logical(bH(kH));
nH  = length(rH);
FaH = GP.gpfa(kH);
FbH = GP.gpfb(kH);
FeH = GP.gpfe(kH)*GP.gpfd;
FdH = GP.gpfd;
BrH = GP.gpbr(kH);
BzH = GP.gpbz(kH);
BaH = GP.gpba(kH);
BnH = zeros(nH,1); BnH(~isnan(BrH) & ~isnan(BzH) | isnan(BaH)) = NaN;
BaH(isnan(BaH)) = 0;
BeH = GP.gpbe(kH)*GP.gpbd;
BdH = GP.gpbd;
CrH = GP.gpcr(kH);
CzH = GP.gpcz(kH);
CaH = GP.gpca(kH);
CnH = zeros(nH,1); CnH(~isnan(CrH) & ~isnan(CzH) | isnan(CaH)) = NaN;
CaH(isnan(CaH)) = 0;
CeH = GP.gpce(kH)*GP.gpcd;
CdH = GP.gpcd;
VrrH = GP.gpvrr(kH);
VrzH = GP.gpvrz(kH);
VzzH = GP.gpvzz(kH);
VeH  = GP.gpve(kH)*GP.gpvd;
VdH  = GP.gpvd;

%% PFC currents
% diagonal part
IaH  = GP.gpia;
IaeH = GP.gpie*GP.gpid;
IadH = GP.gpid;

% linear combinations
IoH   = GP.gpoa;
IoeH  = GP.gpoe*GP.gpod;
IodH  = GP.gpod;
CoH   = L.G.Coa;

%% Passive currents
IuH  = GP.gpua;
IueH = GP.gpue*GP.gpud;
IudH = GP.gpud;

%% PFC dipoles
Dax = L.G.Dda.*GP.gpdw.'; % Assign the correct weights for indiv. coil currents
nda = size(Dax,1);
DaH = GP.gpda;
DeH = GP.gpde*GP.gpdd;
DdH = GP.gpdd;

%% Active voltages
AaH = GP.gpaa;
AeH = GP.gpae*GP.gpad;
AdH = GP.gpad;

%% Equilibrium dependent Green's functions
if     any(~isnan(VrrH) | ~isnan(VrzH) | ~isnan(VzzH) & isfinite(VeH)) % There is an active constraint on the vacuum field curvature
  gsel = {'mut' 'br' 'bz' 'dr1r1mut' 'dr1z1mut' 'dbrdr' 'dbrdz' 'dbzdz'}; gselmode = 0;
elseif any(~isnan( CrH) | ~isnan( CzH) | ~isnan( CnH) & isfinite(CeH)) % There is an active constraint on the poloidal flux hessian
  gsel = {'mut' 'br' 'bz' 'dr1r1mut' 'dr1z1mut'     '0'     '0'     '0'}; gselmode = 1;
elseif any(~isnan( BrH) | ~isnan( BzH) | ~isnan( BnH) & isfinite(BeH)) % There is an active constraint on the poloidal field
  gsel = {'mut' 'br' 'bz'        '0'        '0'     '0'     '0'     '0'}; gselmode = 2;
else
  gsel = {'mut'  '0'  '0'        '0'        '0'     '0'     '0'     '0'}; gselmode = 3;
end

if L.P.MHyinterp
  % compute plasma contribution via interpolation of Mxy - avoids
  % singularities in greenem when rH,zH are on a computational grid point
  MHy = zeros(nH,L.ny); BrHy = MHy; BzHy = MHy;
  VrrHy = MHy; VzzHy = MHy; CrHy = MHy; CzHy = MHy; % init
  % VrrHy and VzzHy are only used as intermediate steps to compute CrHy and CzHy

  % Find control points outside computational grid
  ou = rH>max(L.G.rx) | rH<min(L.G.rx) | zH>max(L.G.zx) | zH<min(L.G.zx);
  % Those can safely be computed with greenem
  [MHy(ou,:),BrHy(ou,:),BzHy(ou,:),CrHy(ou,:),CzHy(ou,:),~    ,~    ,~    ] = greenem(gsel,rH(ou),zH(ou),L.rry ,L.zzy         );

  % control points inside need to use interpolation to be sure
  in = ~ou;
  switch gselmode
    case {0,1}
      [MHy(in,:),BrHy(in,:),BzHy(in,:),VrrHy(in,:),~,~,VzzHy(in,:)] = ...
        qintmex(L.rx,L.zx,L.Mxy,rH(in),zH(in),L.inMfbt);
      CrHy(in,:) = (2*pi*rH(in)).*(VrrHy(in,:) + rH(in).*BrHy(in,:)); % B derivatives to d2psi/drr
      CzHy(in,:) = (2*pi*rH(in)).*(VzzHy(in,:)                     ); % B derivatives to d2psi/drz
    case 2
      [MHy(in,:),BrHy(in,:),BzHy(in,:)] = ...
        qintmex(L.rx,L.zx,L.Mxy,rH(in),zH(in),L.inMfbt);
    case 3
      [MHy(in,:)] = ...
        qintmex(L.rx,L.zx,L.Mxy,rH(in),zH(in),L.inMfbt);
  end
else
  % use greenem
  for iH = 1:numel(rH)
    % check whether any control points are on the computational grid
    assert(~any( (rH(iH) == L.rry(:)) & (zH(iH) == L.zzy(:)) ),...
      'FBT:ControlPointOnGrid',...
      'Control point at [r,z]=[%g,%g] (t=%f) coincides with a grid point. Move the point or use MHyinterp=true',rH(iH),zH(iH),GP.t)
  end
  [MHy,BrHy,BzHy,CrHy,CzHy,~    ,~    ,~    ] = greenem(gsel,rH,zH,L.rry ,L.zzy         );
end

% conductor contributions via Green's functions
[MHa,BrHa,BzHa,CrHa,CzHa,VrrHa,VrzHa,VzzHa] = greenem(gsel,rH,zH,L.G.rw,L.G.zw,L.G.Twa);
[MHu,BrHu,BzHu,CrHu,CzHu,VrrHu,VrzHu,VzzHu] = greenem(gsel,rH,zH,L.G.rv,L.G.zv,L.G.Tvu);

BnHy = sin(BaH).*BrHy      + cos(BaH).*BzHy;
BnHa = sin(BaH).*BrHa      + cos(BaH).*BzHa;
BnHu = sin(BaH).*BrHu      + cos(BaH).*BzHu;

CnHy = atan2(CzHy,CrHy);
CnHa = atan2(CzHa,CrHa);
CnHu = atan2(CzHu,CrHu);
CnHy = sin(CaH).*cos(CnHy) + cos(CaH).*sin(CnHy);
CnHa = sin(CaH).*cos(CnHa) + cos(CaH).*sin(CnHa);
CnHu = sin(CaH).*cos(CnHu) + cos(CaH).*sin(CnHu);

%% Constraints and cost function
% General formulation: (Ax*x ~= b-Ay*Iy) with x = [Ia;Fb] (Note: x will later be rescaled).
% Ax has several components depending on the problem, as discussed below

IDf = @(s,n) repmat(L.dimID.(s),n,1); % function handle to get ID arrays

% Cost function / constraint components
%      PFC current     ; PFC combinations   ; Passive current      PFC dipole          Flux             Br                  Bz                  Bn                  Cr (flux curvature) Cz                  Cn                  Vrr (vacuum dBr/dr)  Vrz                  Vzz
ID  = [IDf('Coils',na) ; IDf('Coil_eqs',no) ; IDf('Passives',nu) ; IDf('Dipoles',nda); IDf('Flux',nH) ; IDf('Fields',nH)  ; IDf('Fields',nH)  ; IDf('Fields',nH)  ; IDf('Hessian',nH) ; IDf('Hessian',nH) ; IDf('Hessian',nH) ; IDf('Vacuum',nH)   ; IDf('Vacuum',nH)   ; IDf('Vacuum',nH)  ];
AIa = [eye(na,na)      ; CoH                ; zeros(nu,na)       ; Dax               ; MHa            ; BrHa              ; BzHa              ; BnHa              ; CrHa              ; CzHa              ; CnHa              ; VrrHa              ; VrzHa              ; VzzHa             ];
AIu = [zeros(na,nu)    ; zeros(no,nu)       ; eye(nu,nu)         ; zeros(nda,nu)     ; MHu            ; BrHu              ; BzHu              ; BnHu              ; CrHu              ; CzHu              ; CnHu              ; VrrHu              ; VrzHu              ; VzzHu             ];
AFb = [zeros(na,1)     ; zeros(no,1)        ; zeros(nu,1)        ; zeros(nda,1)      ; -FbH           ; zeros(nH,1)       ; zeros(nH,1)       ; zeros(nH,1)       ; zeros(nH,1)       ; zeros(nH,1)       ; zeros(nH,1)       ; zeros(nH,1)        ; zeros(nH,1)        ; zeros(nH,1)       ];
AIy = [zeros(na,L.ny)  ; zeros(no,L.ny)     ; zeros(nu,L.ny)     ; zeros(nda,L.ny)   ; MHy            ; BrHy              ; BzHy              ; BnHy              ; CrHy              ; CzHy              ; CnHy              ; zeros(nH,L.ny)     ; zeros(nH,L.ny)     ; zeros(nH,L.ny)    ];
b   = [IaH             ; IoH                ; IuH                ; DaH               ; FaH            ; BrH               ; BzH               ; BnH               ; CrH               ; CzH               ; CnH               ; VrrH               ; VrzH               ; VzzH              ];
err = [IaeH            ; IoeH               ; IueH               ; DeH               ; FeH            ; BeH               ; BeH               ; BeH               ; CeH               ; CeH               ; CeH               ; VeH                ; VeH                ; VeH               ];
errC= [IadH*ones(na,1) ; IodH*ones(no,1)    ; IudH*ones(nu,1)    ; DdH*ones(nda,1)   ; FdH*ones(nH,1) ; BdH*ones(nH,1)    ; BdH*ones(nH,1)    ; BdH*ones(nH,1)    ; CdH*ones(nH,1)    ; CdH*ones(nH,1)    ; CdH*ones(nH,1)    ; VdH*ones(nH,1)     ; VdH*ones(nH,1)     ; VdH*ones(nH,1)    ];

% Separate into constraints and cost function depending on weight:
% Cx*x = d-Cy*Iy, w.*(Ax*x ~= b-Ay*Iy),

ke = ~isfinite(err) | isnan(b); % Remove lines with zero or NaN error or NaN values
kC = ~ke & (err == 0);          % turn lines with zero error into constraints
[AIa,CIa]   = split_cost_constraints(AIa,err,errC,ke,kC);
[AIu,CIu]   = split_cost_constraints(AIu,err,errC,ke,kC);
[AFb,CFb]   = split_cost_constraints(AFb,err,errC,ke,kC);
[AIy,CIy]   = split_cost_constraints(AIy,err,errC,ke,kC);
[b , d  ]   = split_cost_constraints(b  ,err,errC,ke,kC);
IDe = ID(kC);
ID  = ID(~(ke|kC));
w   = 1./err(~(ke|kC));

%% Inequality constraints:
% ** Coil limits **
% Starting from liml < limc*Ia < limu
limu = L.P.limm.*L.P.limu; liml = L.P.limm.*L.P.liml; % Add margin factor to limits

% limits that have equal upper and lower limit values are added directly to equality constraints
j = (limu==liml);
CIa = [CIa;L.P.limc(j,:)]; d  = [d ;limu(j)]; CIy = [CIy;zeros(sum(j),L.ny)];

CiIa = [L.P.limc(~j,:); -L.P.limc(~j,:)];
ci   = [    limu(~j  ); -    liml(~j  )];
CiIy = zeros(2*sum(~j),L.ny  );
CiIu = zeros(2*sum(~j),L.G.nu);
CiFb = zeros(2*sum(~j),     1);

% eliminate inf or invalid constraints
k = all(CiIa==0,2) | any(isnan(CiIa),2) | ~isfinite(ci);
CiIa(k,:) = [];
ci  (k  ) = [];
CiIy(k,:) = [];
CiFb(k,:) = [];
CiIu(k,:) = [];

%% NL method
% group into Ie
 AIe = [AIa,AIu];
CiIe = [CiIa,CiIu];
 CIe = [CIa,CIu];

% Handle different state definitions
if strcmpi(L.P.algoNL,'all-nl')
  % x=[Iy,ag[,Ie]]
   AGS = AIy;
  CiGS = CiIy;
   CGS = CIy;
else % all-nl-Fx, Newton-GS
  % x=[Fx,ag[,Ie]] and Iy = -dlst*Fx(:)./L.rhsf-Tye*Ie
   AGS =      - ( AIy./L.rhsf.')*L.dlst;
  CiGS =      - (CiIy./L.rhsf.')*L.dlst;
   CGS =      - ( CIy./L.rhsf.')*L.dlst;
   AIe =  AIe -   AIy*L.Tye;
  CiIe = CiIe -  CiIy*L.Tye;
   CIe =  CIe -   CIy*L.Tye;
end

if L.P.circuit
  %% Voltage terms (separate equation than ones involving currents)
  Ea = eye(na);
  ke = ~isfinite(AeH) | isnan(AaH); % Eliminate lines with zero or NaN weights or NaN values
  kC = ~ke & (AeH == 0);            % Turn lines with zero weight into constraints
  
  [AVa ,CeVa] = split_cost_constraints( Ea,AeH,AdH*ones(na,1),ke,kC);
  [bVa ,beVa] = split_cost_constraints(AaH,AeH,AdH*ones(na,1),ke,kC);
  wVa = 1./AeH(~(ke|kC));
  
  if L.P.voltlim
    CiVa = [Ea;-Ea];
    biVa = [L.G.Vamax;-L.G.Vamin];
  else
    CiVa = zeros(0,na);
    biVa = [];
  end
  
  %% Express in terms of FGE state and inputs
  % Approximate constraints
  nRa_ = numel(b);
  nRaV = numel(bVa);
  Ra0 = [b;bVa];
  Rax = [[AGS,zeros(nRa_,L.ng),AIe];zeros(nRaV,L.nN)].*L.xscal.';
  Rau = [zeros(nRa_,na);AVa];
  Rab = [AFb;zeros(nRaV,1)];
  w   = [w;wVa];
  ID  = [ID;IDf('Voltage',nRaV)];
  % Equality constraints
  nRe_ = numel(d);
  nReV = numel(beVa);
  Re0 = [d;beVa];
  Rex = [[CGS,zeros(nRe_,L.ng),CIe];zeros(nReV,L.nN)].*L.xscal.';
  Reu = [zeros(nRe_,na);CeVa];
  Reb = [CFb;zeros(nReV,1)];
  IDe = [IDe;IDf('Voltage',nReV)];
  % Inequality constraints
  nRi_ = numel(ci);
  nRiV = numel(biVa);
  Ri0 = [ci;biVa];
  Rix = [[CiGS,zeros(nRi_,L.ng),CiIe];zeros(nRiV,L.nN)].*L.xscal.';
  Riu = [zeros(nRi_,na);CiVa];
  Rib = [CiFb;zeros(nRiV,1)];
else
  %% Assign vessel currents
  masku = (IDe == L.dimID.Passives);
  Iu = IuH; % Prescribed values
  % Update RHS
  b  = b  -  AIe(:,na+1:end)*Iu;
  d  = d  -  CIe(:,na+1:end)*Iu;
  ci = ci - CiIe(:,na+1:end)*Iu;
  % Remove equality constraints
  d(masku) = [];
  CGS(masku,:) = [];
  CIe(masku,:) = [];
  CFb(masku,:) = [];
  IDe(masku) = [];
  
  %% Express in terms of FGS state and inputs
  % Approximate constraints
  nRa = numel(b);
  Ra0 =  b;
  Rax = [AGS,zeros(nRa,L.ng)].*L.xscal.';
  Rau =  AIe(:,1:na);
  Rab =  AFb;
  % Equality constraints
  nRe = numel(d);
  Re0 =  d;
  Rex = [CGS,zeros(nRe,L.ng)].*L.xscal.';
  Reu =  CIe(:,1:na);
  Reb =  CFb;
  % Inequality constraints
  nRi = numel(ci);
  Ri0 =  ci;
  Rix = [CiGS,zeros(nRi,L.ng)].*L.xscal.';
  Riu =  CiIe(:,1:na);
  Rib =  CiFb;
end

%% Bounding rectangle
bb = meqbbox(rH(bH),zH(bH),L.G.rl,L.G.zl,idoublet);

end

function [A,C] = split_cost_constraints(A,err,errC,ke,kC)
kk = ~(ke|kC); % keep in cost
C = (1./errC(kC)).*A(kC,:);
A = (1./err (kk)).*A(kk,:);
end
