function X = fbtgp(X,varargin)
% X = fbtgp(X,r,z,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve,timeder)
%%
% Appends entry to the set of gp*/g1*/g2* variables describing geometry constraints
% for FBT and their time derivatives. For details of the equations, see FBTHELP
%
% X: previous structure
% r,z: r,z location of point(s) to add
% b  : Point is on main plasma boundary (used only for initial guess of current distribution)
% fa : Flux value
% fb : if 0, then fa is the absolute flux value; if 1, then fa is defined relative to an unknown offset common to all points with fb=1
% fe : Weight of flux constraint (multiplies fd)
% br : Value of radial magnetic field
% bz : Value of vertical magnetic field
% ba : Angle of magnetic field
% be : Weight of magnetic field constraint (multiplies bd)
% cr : Value of d2/dr2(Psi)
% cz : Value of d2/drdz(Psi)
% ca : [To be clarified] Angle of hessian of Psi
% ce : Weight of hessian of Psi constraints (multiplies cd)
% vrr: Value of d/dr(Br), vacuum field only
% vrz: Value of d/dr(Bz), vacuum field only
% vzz: Value of d/dz(Bz), vacuum field only
% ve : Weight of vacuum field derivative constraints (multiplies vd)
%
% timeder: time derivative order of the constraint (default=0)
%   td=0/1/2 populate gp*/g1*/g2* variables respectively
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% initialize
if ~isfield(X,'t')
  X.t = 0; %default
end
nt = numel(X.t);

list = {'r','z','b','fa','fb','fe','br','bz','ba','be','cr','cz','ca','ce','vrr','vrz','vzz','ve'};
assert(numel(varargin)>=numel(list),'fbtgp expects at least %d+1 arguments, found %d+1',numel(list),numel(varargin))

if numel(varargin) == numel(list)+1
  td = varargin{end}; % order of time derivative
  assert(td>=0 && td <= 2 && round(td)==td,'td must be a scalar between 0 and 2')
else
  td = 0; % default
end

% gp* / g1* / g2* depending on 0th, 1st, 2nd time derivative
switch td
  case 0
    gp = 'gp';
  case 1
    gp = 'g1';
  case 2
    gp = 'g2';
end

gplist = strcat(gp,list);

% Initialize nonexistent fields
for var = gplist
  if ~isfield(X,var{1})
    X.(var{1}) = NaN(0,nt);
  end  
end

% add global weights if any are missing
list = strcat(gp,{'fd','bd','cd','id','dd','vd'});
for var = list
  if ~isfield(X,var{1})
    X.(var{1}) = Inf(1,nt);
  end
end

% Skip if r and z are NaN
if ~any(~isnan(varargin{1}) & ~isnan(varargin{2}))
  return
end

[m,n] = size(varargin{1}); % gpr size
for ii = 1:numel(gplist)
  myfield = gplist{ii};
  if myfield(end)=='e' % weights
    default = Inf;
  elseif myfield(end)=='b' % gpb/gpfb
    default = false;
  else
    default = NaN;
  end
  X.(myfield) = gpcat(X.(myfield),varargin{ii},m,n,default);
end
end

function c = gpcat(a,b,m,n,d)
% concatenate gp entries
% X = gpcat(X,entries,m,n)
% m: number of rows, n: number of columns
% d: default value
if isempty(b),      b = repmat(d,m,n);
elseif isscalar(b), b = repmat(b,m,n);
end
assert(isempty(a) || size(a,2)==size(b,2),'FBTE:Incompatible number of equilibria when converting geometrical parameters')
c = [a;b];
end
