%FBTC  FBTE ancillary data
% L = FBTC(P,G) returns a structure with FBTE ancillary data. P contains
% the configuration parameters as obtained by FBTP<TOK>. G contains the
% geometry as obtained by FBTG<TOK>. See also FBTP,FBTG.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
function L = fbtc(P,G,varargin)

L = meqc(P,G);

%% Code name
L.code = 'fbt';

%% Constraint functions
L.agconc = meqagconc(P.fbtagcon,L.TDg,P.idoublet(1)); % consolidate constraint equations
L.nC     = size(L.agconc,1);                          % number of constraint equations

%% CDE
L.cdec = {};
L.np = 0;
assert(L.nC + L.np == L.ng,...
  'Number of ag constraints (%d) and CDE residuals (%d) must match number of basis functions (%d)',...
  L.nC,L.np,L.ng)

%% NL model setup
[L.ind,L.nN,L.dimN,L.xscal,L.resscal,L.LX2x,L.isEvolutive,L.nuN,L.nrD] = meqalgoNLc(L,P.algoNL,P.circuit);

%% Setup for analytical jacobian
% Additional derivative functions
if P.gsxe~=3, [L.Mxy,L.Mxe] = meqFxjac(L); end

%% Define matrices for temporal evolution problem (from fgec.m)
if P.circuit
  Mya = G.Mxa(L.lxy(:),:); % Mutual between internal grid and active coils
  Myu = G.Mxu(L.lxy(:),:); % Mutual between internal grid and vessel segments

  L.Mey = [Mya.' ; Myu.']; % Mutual between external currents and plasma current

  L.Mee = [G.Maa , G.Mau; G.Mau.', G.Muu]; % Mutual inductance between external currents
  L.Re  = [G.Ra;G.Ru]; % Resistance of external currents
end

%% Linear part of jacobians
[L.Jx,L.Ju,L.Jxdot] = fgeFvacJac(L);

%% Optional Jacobian of computational grid flux vs currents
if P.MHyinterp
  L.inMfbt = qintc(P.inp,L.drx,L.dzx);
end

%% Checks
assert(size(P.limc,2)==G.na,'limc size does not match G.na')

%% Identifier strings for cost function terms
IDnames = {'Coils','Coil_eqs','Dipoles','Passives','Flux','Fields','Hessian','Vacuum','Voltage'};
L.dimID = cell2struct(num2cell(1:numel(IDnames))',IDnames);
end
