function M = dlstc(nzx,nrx,drx,dzx,rry)
% DLSTC assembles the sparse operator matrix of the GS operator
% function M = dlstc(drx,dzx,rry)
% Arguments:
%   nzx: int, number of x-grid elements along Z-axis
%   nrx: int, number of x-grid elements along R-axis
%   drx: float, R-axis grid spacing
%   dzx: float, Z-axis grid spacing
%   rry: Matrix, Matrix containing R values for y-grid
% returns:
%   M:   Sparse Matrix, (ny, nx)-matrix representing the dlst operator
%
% For details, see: [J-M.Moret et al. Fus.Eng.Des 2015] Section 4.2
%              and ﻿[F. Hofmann, Comput. Phys. Commun. 48(2) 1988].
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

A = (dzx/drx)^2*rry./(rry + drx/2);
B = (dzx/drx)^2*rry./(rry - drx/2);
C = 2 + A + B;

% compute additional index quantities
nzy = nzx-2; nry = nrx-2;
nx = nzx*nrx; ny = nzy*nry;

% getting all y grid indices and adding one to make them the x grid indices
iy = 1:ny;
[izy, iry] = ind2sub([nzy, nry], iy);
ix = sub2ind([nzx,nrx], izy+1, iry+1);

% combining all indices as well as the coeffs of dlst into big vectors
inds_in    = [ix   , ix+1       , ix-1       , ix+nzx, ix-nzx];
inds_out   = [iy   , iy         , iy         , iy    , iy    ];
vals_diags = [-C(:); ones(ny, 1); ones(ny, 1); A(:)  ; B(:)  ];

% constructing the sparse operator
M = sparse(inds_out, inds_in, vals_diags, ny, nx);
end
