%BFSP  B-spline basis functions
%
% Basis-functions are constructed using order 5 B-splines (degree 4) for
% the integral functions and their exact derivatives for p' and TT'. The
% knot distribution is so far the same for basis functions associated to p'
% or TT'. The parameters P must be obtained by calling the BFPSP function.
%
% No regularisation has been implemented yet.
%
% See also: BFPSP, BFCT, BFHELP.
%
% For details see [MEQ-redbook]
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

function varargout = bfsp(mode,P,varargin)

% Handle parameters
nP = size(P.MP,2); nT = size(P.MT,2);
fPg = [ ones(nP,1) ; zeros(nT,1)];
fTg = [zeros(nP,1) ;  ones(nT,1)];

% Prepare structure for bfct
Pbfct.name = 'bfsp';
Pbfct.fPg  = fPg;
Pbfct.fTg  = fTg;
Pbfct.f    = [];
Pbfct.fN   = @(FQ) bfsp_fN(P,FQ);
Pbfct.fag  = [];
Pbfct.fA   = [];
Pbfct.df   = [];
Pbfct.dfN  = @(FQ) bfsp_dfN(P,FQ);
Pbfct.dfag = [];
Pbfct.dfA  = [];

% Call generic bfct
[varargout{1:nargout}] = bfct(mode,Pbfct,varargin{:});
end

function [gQ,IgQ] = bfsp_fN(P, FQ)
% BFSP_FN Compute BFSP normalized basis functions

% Compute normalized basis functions for fPg
 GP = bspsum(P.tauP,P.MP,FQ(:),1);
IGP = bspsum(P.tauP,P.MP,FQ(:),0);
% Compute normalized basis functions for fTg
 GT = bspsum(P.tauT,P.MT,FQ(:),1);
IGT = bspsum(P.tauT,P.MT,FQ(:),0);
% Dispatch to 1...ng
 gQ = [ GP, GT];
IgQ = [IGP,IGT];
end

function [dgQ,dIgQ] = bfsp_dfN(P, FQ)
% BFSP_DFN Compute derivatives of BFSP normalized basis functions

% Compute derivatives of normalized basis functions for fPg
 dGP = bspsum(P.tauP,P.MP,FQ(:),2);
dIGP = bspsum(P.tauP,P.MP,FQ(:),1);
% Compute derivatives of normalized basis functions for fTg
 dGT = bspsum(P.tauT,P.MT,FQ(:),2);
dIGT = bspsum(P.tauT,P.MT,FQ(:),1);
% Dispatch to 1...ng
 dgQ = [ dGP, dGT];
dIgQ = [dIGP,dIGT];
end
