function [rint,zint] = ray_trace(r,z, r0, z0, theta, varargin)
%RAY_TRACE Compute intersection of ray starting at r0,z0 intersecting the contour described in r,z
%
% Inputs:
% r: [m] 1darray containing R coordinate of contour
% z: [m] 1darray containing Z coordinate of contour
% r0: [m] R coordinate of origin of the ray
% z0: [m] Z coordinate of origin of the ray
% theta: (rad) angle in radiant for the direction of the ray.
%
% Outputs:
% rint,zint: 1darrays, containing all intersection of the ray with the contour describe in r,z
%
% The ray is described by the equation zr = (xr-xr0)tr + zr0
% where tr is the tan(theta)
% The geometry outline in r,z is assumed to be a sequence of linear segments.
% The segment is described by the line zs = (xs - xs0)ts + zs0
% where ts = (zs2 - zs1)/(rs2 - zs1) being (rs1,zs1),(rs2,zs2) the
% coordinates of the segment.
% The algothm loop over all segments, finds the intersection with the
% ray where xr = xs, yr = ys, and check if the intersection is part of the segment.
% The algorithm return all intersecting points of a given ray, not just the first one.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Get parameters
P = struct;
for k = 1:2:length(varargin), P.(varargin{k}) = varargin{k+1}; end

% Extra parameters (not exposed)
drthr_vertical = 1e-5; % [m] lower limit of delta r to consider to point belonging to a vertical segment

%% Loop over all segments
rint = []; zint = [];
for ii = 1:numel(r)-1
  % Extract the segment
  rs1 = r(ii); zs1 = z(ii);
  rs2 = r(ii+1); zs2 = z(ii+1);
  ts = (zs2-zs1)/(rs2-rs1);
  tr = tan(theta);
  
  if abs(rs2 - rs1)<drthr_vertical % Special case vertical segments
    rhat = rs1;
    zhat = (rhat - r0)*tr + z0;
  else % Compute intersection ray segment
    A = [tr , -1; ts  ,-1];
    b = [r0*tr - z0; rs1*ts-zs1];
    
    sol = A\b;
    rhat = sol(1);
    zhat = sol(2);
  end
  
  % Debugging plot
  if isfield(P, 'debugplot') && P.debugplot > 2
    figure
    plot(rhat,zhat,'ro','DisplayName', 'intersection' )
    plot([rs1,rs2],'DisplayName', 'segment')
    title(sprintf('Index %d', ii))
    xlabel('R')
    ylabel('Z')
    legend
  end
  
  % Check if interesection point belongs to the segment
  rmin = min([rs1, rs2]); zmin = min([zs1, zs2]);
  rmax = max([rs1, rs2]); zmax = max([zs1, zs2]);
  if rhat >= rmin && rhat <= rmax && zhat >= zmin && zhat <= zmax
    rint = [rint; rhat]; zint = [zint; zhat];
  end
end

end
