function G = pf_active2G(G, pf_active, P)
%PF_ACTIVE2G Extract G structure form ids.pf_active
%
% The IDS tree has the following naming convention
% Power supplies
% Coils are groups of serially connected windings.
% Coils can be sub-diveded into elements.
% Circuits are groups of coils serially connected attached to a power supply.
% Coils in circuits are typically all series connected.
% Power supplies together with attached Coils = Circuits.
%
% Active pf circuits as in MEQ naming convention
% Iw current flowing in windings. Windings are serially connected.
% Ic current flowing in individual coil corresponding to pf_active.coils in ids
% Ia current flowing in a series connected circuits corresponding to pf_active.circuit
% Twa = from active circuit to windings Iw = G.Twa*Ia. Twa is usually a 0/1 or 0/-1 matrix, but can be float in to take into account partial toroidal windings.
% Twc = from indivial coil to windings Iw = G.Twc * Ic. Twa is usually a 0/1 matrix. Must be positive!. It can be float to take into account partial toroidal windings.
% Tca = from active circuit to individual coil Ic = Tca*Ia therefore Twa = Twc*Tca. Tca must only take values in the list [-1,0,1].
% These matrix is used to group together windings belonging to the same coil. Windings of 1 coil should have the same current.
% Matrix Tca is composed of columns of 1/0 to select which coil belongs to a given active circuit.
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% Overwrite defaults
D = struct;
D.debug = 0;
D.debugplot = 0;
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

%%  Interpreting connection matrix to extract relation between coils and circuits
if isempty(pf_active)
  error('input pf_active is empty')
end
info_connection_matrix = extract_info_connection_matrix(pf_active);

% Construct the grouping matrix coils to windings Twc.
% This has to be constructed dinamically because every coil can have a
% different type of representation in principle.
G.rw = []; G.zw = []; G.Twc = []; G.Rw = []; G.aw = []; G.ww = []; G.hw = []; G.dw = [];
G.rc = []; G.zc = []; G.wc = []; G.hc = [];
signs = [];
for ii = 1:info_connection_matrix.ntotcoils % loop over coils
  Twc_ii = []; rw_ii = []; zw_ii = []; aw_ii = []; ww_ii = []; hw_ii = []; dw_ii=[];

  % Get the sign of the first element of a coil. All elements must have the same sign.
  signs(ii) = sign(info_connection_matrix.coil{ii}.element{1}.turns_with_sign);
  for jj = 1:numel(info_connection_matrix.coil{ii}.element)% loop over elements of each coil
    switch  info_connection_matrix.coil{ii}.element{jj}.geometry.geometry_type
      case 1 % outline
        error('geometry_type outline not yet implemented')
        % Implementation in wall2G could be re-used
      case 2 % rectangle
        pfrzdrdz = zeros(4,1);
        pfrzdrdz(1) = info_connection_matrix.coil{ii}.element{jj}.geometry.rectangle.r;
        pfrzdrdz(2) = info_connection_matrix.coil{ii}.element{jj}.geometry.rectangle.z;
        pfrzdrdz(3) = info_connection_matrix.coil{ii}.element{jj}.geometry.rectangle.width;
        pfrzdrdz(4) = info_connection_matrix.coil{ii}.element{jj}.geometry.rectangle.height;
        % All element of a given coil must have the same sign
        if sign(info_connection_matrix.coil{ii}.element{jj}.turns_with_sign) ~= signs(ii)
          error('All elements of a coil must have the same sign')
        end
        turns_with_sign =  abs(info_connection_matrix.coil{ii}.element{jj}.turns_with_sign);
        [rw_jj, zw_jj, aw_jj, ww_jj, hw_jj] = gen_filament(pfrzdrdz, turns_with_sign, 2);

        % The discretization generated might have number of windings ~= number of
        % turns. In order to conserve the total current flowing through the poloidal
        % plane Twc provides an effective number of turns.
        nrw = numel(rw_jj);
        Twc_jj = zeros(nrw,1);
        Twc_jj(:,1) = 1*abs(turns_with_sign/nrw);

        rw_ii = [rw_ii;rw_jj]; zw_ii = [zw_ii;zw_jj];
        aw_ii = [aw_ii;aw_jj]; ww_ii = [ww_ii;ww_jj];
        hw_ii = [hw_ii;hw_jj]; Twc_ii = [Twc_ii; Twc_jj];
        dw_ii = [dw_ii;nan(nrw,1)];
      case 3 % oblique
        error('geometry_type oblique not implemented')
      case 4 % arcs circle
        error('geometry_type outline not implemented')
      case 5 % annulus is consider to be a single filaments with multiple effective turns
        rw_ii(end+1,1)  = info_connection_matrix.coil{ii}.element{jj}.geometry.annulus.r;
        zw_ii(end+1,1)  = info_connection_matrix.coil{ii}.element{jj}.geometry.annulus.z;
        Twc_ii(end+1,1) = abs(info_connection_matrix.coil{ii}.element{jj}.turns_with_sign);
        aw_ii(end+1,1) = info_connection_matrix.coil{ii}.element{jj}.area;
        ww_ii(end+1,1) = sqrt(aw_ii(end,1));
        hw_ii(end+1,1) = ww_ii(end,1);
        dw_ii(end+1,1) = NaN; % TODO. Circular section not implemented yet.
    end % End of element loop
  end % End of coil loop
  
  G.rw = [G.rw; rw_ii]; G.zw = [G.zw; zw_ii];
  G.aw = [G.aw; aw_ii]; G.ww = [G.ww; ww_ii];
  G.hw = [G.hw; hw_ii]; G.dw = [G.dw; dw_ii];
  G.rc = [G.rc; (max(rw_ii)+min(rw_ii))/2 ]; G.zc = [G.zc; (max(zw_ii)+min(zw_ii))/2];
  G.wc = [G.wc; (max(rw_ii+ww_ii/2)-min(rw_ii-ww_ii/2))]; G.hc = [G.hc; (max(zw_ii+hw_ii/2)-min(zw_ii-hw_ii/2))];
  G.Twc(end+1:end+numel(Twc_ii(:)),end+1) = Twc_ii(:);
  
  Rc = info_connection_matrix.coil{ii}.resistance;
  
  % Given that windings are serially connected.
  % Iw = Twc*Ic
  % Take Vw = diag(Rw)*Iw
  % Vc = sum(Vw) = Rc*Ic = Twc'*Vw = (Twc'diag(Rw)Twc)*Ic
  % Hence
  % Rc = Twc'*diag(Rw)*Twc
  % Rw = 2*pi*rho*rw/aw

  % Compute resistivity for the coil assuming homogeneity.
  multiplier = 2*pi*Twc_ii'*diag(rw_ii./aw_ii)*Twc_ii;
  rho = Rc/multiplier;

  % Compute the resistance of each winding.
  G.Rw = [G.Rw; 2*pi*rho*rw_ii./aw_ii];
end

G.na = info_connection_matrix.ntotcircuits;
for ii=1:G.na
  G.dima{ii,1} = info_connection_matrix.circuit{ii}.name;
end

% Build Tca.
for ii=1:G.na
  for jj=1:info_connection_matrix.ntotcircuits
    if strcmp(info_connection_matrix.circuit{jj}.name, G.dima{ii})
      for kk=info_connection_matrix.circuit{jj}.coils_ind_belonging_to_circuit
        % All element of a given coil must have the same sign. This is checked before.
        G.Tca(kk, ii) = sign(info_connection_matrix.coil{kk}.element{1}.turns_with_sign);
      end
    end
  end
end
G.Twa = G.Twc*G.Tca;

%% Limits from supplies
for ii = 1:info_connection_matrix.ntotsupplies % loop on coil
  if isfield(info_connection_matrix.supply{ii}, 'voltage_limit_max')
    G.Vamax(ii) = info_connection_matrix.supply{ii}.voltage_limit_max;
    G.Vamin(ii) = info_connection_matrix.supply{ii}.voltage_limit_min;
  else
    G.Vamax(ii) = +Inf;
    G.Vamin(ii) = -Inf;
  end
end

%% TODO. Read coil limits from IDS.
G.Iamax = +Inf(G.na,1);
G.Iamin = -Inf(G.na,1);
G.Talim =  Inf(G.na,1);

if P.debugplot>2
  figure
  title('Winding representation')
  meqgplot(G, gca, 'w')
end

end
