function connections = G2connections(G)
%G2connectins convert G structure to connection matrix in IDS format
% 
% Description of the connection matrix is found in IDS structure pf_active.circuit(ii).connections
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

% The following is a copy of the documentation taken from IDS version = 3.37.0
% Description of the supplies and coils connections (nodes) across the circuit. 
% Nodes of the circuit are listed as the first dimension of the matrix.
% Supplies (listed first) and coils (listed second) SIDES are listed as the second dimension.
% Thus the second dimension has a size equal to 2*(N_supplies+N_coils). 
% N_supplies (resp. N_coils) is the total number of supplies (resp. coils) listed in the supply (resp.coil) array of structure,
% i.e. including also supplies/coils that are not part of the actual circuit.
% The (i,j) matrix elements are 1 if the j-th supply or coil side is connected to the i-th node, or 0 otherwise.
% For coils, sides are listed so that a current flowing from side 1 to side 2 (inside the coil) is positive
% (i.e. counter-clockwise when seen from above)

connections = {};
assert(all(isfield(G,{'na','Tca','dimc'})), 'G must contain na, Tca and dimc')
assert(all(any(G.Tca,2)), 'All active circuits must have at least 1 coil')
assert(all(sum(abs(G.Tca),2)==1), 'All coils must belong to a single active circuit')
assert(all(ismember(G.Tca(:),[-1,0, 1])), 'Tca entries must be [-1,0,1]')

for ia=1:G.na % Iterator on active circuits
  ind_c_ia = find(G.Tca(:,ia)~=0);
  nc = numel(G.dimc);
  nc_ia = numel(ind_c_ia); % Number of coils belonging to circuit ia
  n_nodes = nc_ia + 1; % Number of nodes in the circuit
  connection = zeros(n_nodes, 2*nc + 2*G.na);

  % Initialize the power supply part of the connection matrix of a given circuit
  connection(1,2*ia-1) = 1;
  connection(n_nodes,2*ia) = 1;
  for jj=1:nc_ia % Iterator on coils
    ic = ind_c_ia(jj);
    signc = sign(G.Tca(ic,ia));
    if signc > 0 % Normal connection
      connection(jj, 2*G.na + 2*ic-1) = 1;
      connection(jj+1, 2*G.na + 2*ic) = 1;
    else % Reverse connection. (Usually for vertical stability coils)
      connection(jj, 2*G.na + 2*ic) = 1;
      connection(jj+1, 2*G.na + 2*ic-1) = 1;
    end
  end
  connections{ia} = connection;
end
end