function [psi,Br,Bz,absBp,H,gradB,M] = GS_vacuum_analytic(c,d,scal)
% GS_VACUUM_ANALYTIC returns anonymous functions for vacuum analytical GS solutions
%
% [psi,Br,Bz,absBp,H,gradB,M] = GS_homogeneous(c,d,scal)
% A parametrized analytical vacuum solution of GS equation.
% Returns anonymous functions to evaluate key quantities.
%
% From R Srinivasan paper PPCF 52 (2010) 035007, (eq.11)
%
% Inputs:
%   c: vector (5x1) of coefficients for symmetric part
%   d: vector (4x1) of coefficients for asymmetric part
%   scal: global scaling factor
%
% Outputs: anonymous functions for
%  psi (0th order)
%  Br, Bz, |Bp| (1st order)
%  H = [psirr,psirz;psizr,psizz] (Hessian, 2nd order derivatives)
%  gradB = [Brr,Brz;Bzr,Bzz]; (Local field gradient)
%  M = 3D Matrix containing Jacobian of the Hessian (3rd order derivatives)
%  If multiple R,Z points are provided, these go in the last dimension 
%
% Examples:
%   c = zeros(5,1); d=zeros(4,1); s=1;
%   c(2) = 3; c(5) = -2; d(2) = -3; % coefficientss
%   Rgrid = linspace(0.7,0.9,101);
%   Zgrid = linspace(-0.2,0.2,81);
%   [psi,Br,Bz,absBp,H,gradB] = GS_vacuum_analytic(c,d,s)
%   [RR,ZZ] = meshgrid(Rgrid,Zgrid);
%   subplot(121); contourf(RR,ZZ,psi(RR,ZZ)); axis equal tight;
%   subplot(122); contourf(RR,ZZ,absBp(RR,ZZ)); axis equal tight;
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

%% Analytical flux and fields and derivatives
[psi,dpsidr,dpsidz,psirr,psizz,psirz,psirrr,psizzz,psirrz,psirzz] = GS_analytic(c,d,scal);

%% Other anonymous function for derived quantities
% COCOS=17
Br = @(R,Z) -dpsidz(R,Z)./(2*pi*R);
Bz = @(R,Z)  dpsidr(R,Z)./(2*pi*R);

absBp = @(R,Z) sqrt(Br(R,Z).^2 + Bz(R,Z).^2);

dBrdr = @(R,Z) -1./(2*pi) * (psirz(R,Z)./R - dpsidz(R,Z)./R.^2);
dBzdr = @(R,Z)  1./(2*pi) * (psirr(R,Z)./R - dpsidr(R,Z)./R.^2);

dBrdz = @(R,Z) -1./(2*pi*R) .* psizz(R,Z);
dBzdz = @(R,Z)  1./(2*pi*R) .* psirz(R,Z);

rs3 = @(x) reshape(x,1,1,numel(x));
gradB =  @(R,Z) [dBrdr(rs3(R),rs3(Z)),dBrdz(rs3(R),rs3(Z));dBzdr(rs3(R),rs3(Z)),dBzdz(rs3(R),rs3(Z))];

H = @(R,Z) [psirr(rs3(R),rs3(Z)),psirz(rs3(R),rs3(Z));psirz(rs3(R),rs3(Z)),psizz(rs3(R),rs3(Z))];

rs4 = @(x) reshape(x(:),1,1,1,numel(x));
M = @(R,Z) cat(3,...
  [psirrr(rs4(R),rs4(Z)),psirrz(rs4(R),rs4(Z));
   psirrz(rs4(R),rs4(Z)),psirzz(rs4(R),rs4(Z))],...
  [psirrz(rs4(R),rs4(Z)),psirzz(rs4(R),rs4(Z));
   psirzz(rs4(R),rs4(Z)),psizzz(rs4(R),rs4(Z))]); % monkey saddle: 3D tensor as Jacobian of the Hessian per point
end

function [psi,dpsidr,dpsidz,psirr,psizz,psirz,psirrr,psizzz,psirrz,psirzz] = GS_analytic(c,d,s)

assert(numel(c)==5,'c must have 5 elements');
assert(numel(d)==4,'d must have 4 elements');

% Expressions constructed using scripts/GS_symbolic.m

psi    = @(R,Z) s*(c(1) + c(2)*(R.^2) + c(3)*(R.^4 - 4*R.^2.*Z.^2) + c(4)*(R.^2.*log(R) - Z.^2) ...
  + c(5)*(R.^6 + 8*R.^2.*Z.^4 - 12*R.^4.*Z.^2) ...
  + d(1)*(Z) + d(2)*(R.^2.*Z) + d(3)*(3*R.^4.*Z - 4*R.^2.*Z.^3) ...
  + d(4)*(3*R.^2.*Z.*log(R) - Z.^3));

dpsidr = @(R,Z) s*(2*R*c(2) + c(5)*(6*R.^5 - 48*R.^3.*Z.^2 + 16*R.*Z.^4) - c(3)*(- 4*R.^3 + 8*R.*Z.^2) + c(4)*(R + 2*R.*log(R)) + d(4)*(3*R.*Z + 6*R.*Z.*log(R)) - d(3)*(- 12*R.^3.*Z + 8*R.*Z.^3) + 2*R.*Z*d(2));
dpsidz = @(R,Z) s*(d(1) - d(4)*(3*Z.^2 - 3*R.^2.*log(R)) - 2*Z*c(4) - c(5)*(24*R.^4.*Z - 32*R.^2.*Z.^3) + R.^2*d(2) + d(3)*(3*R.^4 - 12*R.^2.*Z.^2) - 8*R.^2.*Z*c(3));
psirr  = @(R,Z) s*(2*c(2) + c(5)*(30*R.^4 - 144*R.^2.*Z.^2 + 16*Z.^4) + 2*Z*d(2) + c(4)*(2.*log(R) + 3) + c(3)*(12*R.^2 - 8*Z.^2) + d(4)*(9*Z + 6*Z.*log(R)) + d(3)*(36*R.^2.*Z - 8*Z.^3));
psizz  = @(R,Z) s*(- 2*c(4) - 6*Z*d(4) - 8*R.^2*c(3) - c(5)*(24*R.^4 - 96*R.^2.*Z.^2) - 24*R.^2.*Z*d(3));
psirz  = @(R,Z) s*(2*R*d(2) + d(4)*(3*R + 6*R.*log(R)) - d(3)*(- 12*R.^3 + 24*R.*Z.^2) + c(5)*(- 96*R.^3.*Z + 64*R.*Z.^3) - 16*R.*Z*c(3));
psirrr = @(R,Z) s*( 24*R*c(3) + (2*c(4))./R - c(5)*(288*R.*Z.^2 - 120*R.^3) + 72*R.*Z*d(3) + (6*Z*d(4))./R);
psizzz = @(R,Z) s*(192*R.^2.*Z*c(5) - 24*R.^2*d(3) - 6*d(4));
psirrz = @(R,Z) s*(2*d(2) - 16*Z*c(3) + d(4)*(6*log(R) + 9) + d(3)*(36*R.^2 - 24*Z.^2) - c(5)*(288*R.^2.*Z - 64*Z.^3));
psirzz = @(R,Z) s*(c(5)*(192*R.*Z.^2 - 96*R.^3) - 16*R*c(3) - 48*R.*Z*d(3));

end

