classdef mds_basic_tests < matlab.unittest.TestCase
  % a simple parametrized test for mds
  % see documentation of matlab.unittest suite for more details!
  
  
  properties
    Shot = 40000;
    Tree = 'tcv_shot';
    DefaultServer = 'tcvdata.epfl.ch'; % default server to try direct connection
    Server; % final server for mdsconnect (determined later)
    verbosity = false;
  end
  
  properties(TestParameter)
    config   = {'remote','local'}
    datatype = {'int8','uint8','int16','uint16','int32','uint32','single'}
    array    = {false,true}
  end
  
  methods(TestClassSetup)
    function temporary_working_folder(testCase)
      testCase.assertTrue(~verLessThan('matlab','8.2'),'This test suite is incompatible with MATLAB 8.1 or earlier');
      if ~verLessThan('matlab','9.0')
        testCase.applyFixture(matlab.unittest.fixtures.WorkingFolderFixture);
      else
        tempFixture = testCase.applyFixture(matlab.unittest.fixtures.TemporaryFolderFixture);
        testCase.addTeardown(@() cd(pwd));
        cd(tempFixture.Folder);
      end
      fprintf('Created folder %s as temporary working folder\n',pwd);
    end

    function check_server_options(testCase)
      % check default server

      % check if we can mdsconnect to it
      fprintf('Attempt to mdsconnect to %s\n',testCase.DefaultServer);
      success = mdsconnect(testCase.DefaultServer);
      if success % all ok
        testCase.Server = testCase.DefaultServer;
        return
      else
        fprintf('Could not mdsconnect to %s',testCase.DefaultServer);
      end
      
    end

    function clear_mdsipmex(~)
      mdsdisconnect('all'); pause(0.05);
    end
  end
  
  methods(Test)
    
    function test_mdsconnect(testCase)
      % Verify mdsconnect without any argument is successful
      status=mdsconnect();
      testCase.verifyEqual(status,1,'mdsconnect without any argument was not successful');
      mdsdisconnect;

      % Verify connection to default server is successful
      status=mdsconnect(testCase.Server);
      testCase.verifyEqual(status,1,'Connection to default server was not successful');
      mdsdisconnect;

      % Verify connection to dummy server fails
      status=mdsconnect('dummy.mds.ip');
      testCase.verifyEqual(status,0,'Connection to dummy server did not fail');
    end

    function test_mdscurrent(testCase)
      % Verify mdscurrent returns name of current connection
      mdsconnect(testCase.Server);
      server=mdscurrent;
      testCase.verifyEqual(server,testCase.Server,'mdscurrent return value does not match default server name');
    end

    function test_mdsdisconnect(testCase)
      % Verify mdsdisconnect leaves us with no active connection
      mdsconnect(testCase.Server);
      status=mdsdisconnect;
      testCase.verifyEqual(mod(status,2),1,'mdsdisconnect failed')
      testCase.verifyEqual('',mdscurrent,'mdscurrent should return empty after mdsdisconnect');

      % Verify mdsdisconnect('all') leaves an empty connection list
      status1=mdsconnect(testCase.Server);
      status2=mdsconnect('spcsrv1.epfl.ch');
      testCase.verifyEqual(status1*status2,1,'One of the 2 connections failed');
      status=mdsdisconnect('all');
      testCase.verifyEqual(mod(status,2),1,'mdsdisconnect(''all'') failed')
      testCase.verifyEqual(0,mdsremotelist,'mdsremotelist should return 0 after mdsdisconnect(''all'')');
    end
    
    function test_datatypes(testCase,config,datatype,array)
      % check that both value and valueraw work as expected
      
      switch config
        case 'remote'
          status=mdsconnect(testCase.Server);
        case 'local'
          status=mdsconnect('local');
      end
      testCase.verifyEqual(status,1,sprintf('Connection to %s server failed',config));
      
      if array, dims = [4,3];
      else,     dims = [1,1];end
      
      s = rng(1);
      data = randi(12,dims,datatype);
      rng(s);
      
      value = mdsvalue('$',data);
      testCase.verifyClass(value,'double');
      
      valueraw = mdsvalueraw('$',data);
      testCase.verifyClass(valueraw,datatype);
      
      testCase.verifyEqual(value,double(valueraw));
    end
    
    function test_mdsopen(testCase)
      
      mdsconnect(testCase.Server);
      [shoto,stat] = mdsopen(testCase.Shot);
      testCase.verifyTrue(mod(stat,2)==1,'mdsopen returned odd status');
      testCase.verifyEqual(shoto,testCase.Shot);
      mdsclose;
      
      [shoto,stat] = mdsopen(testCase.Tree,testCase.Shot);
      testCase.verifyTrue(mod(stat,2)==1,'mdsopen returned odd status');
      testCase.verifyEqual(shoto,testCase.Shot);
      mdsclose;
    end
    
    function test_mdsvalue(testCase)
      mdsconnect(testCase.Server);
      mdsopen(testCase.Tree,testCase.Shot);
      
      test_expression = '\magnetics::iplasma:trapeze';
      mdsvalue(test_expression);
      
      test_expression = 'dim_of(\magnetics::iplasma:trapeze)';
      mdsvalue(test_expression);
      
      mdsclose;
    end
    
    function test_mdsput(testCase)
      testCase.assumeFail('Ignore mdsput test for now pending better implementation by IT group')
      
      % Below, find a stub for this test
      % write to local tree on Cristian Galperti's machine (for testing)
      server = 'crpppc171.epfl.ch';
      tree = 'tt1';
      node = 'FORCEON';
      
      testCase.assumeTrue(0==system(sprintf('ping -c1 %s > /dev/null',server)),sprintf('Could not ping %s',server));
      
      try
        mdsconnect('crpppc171');
        mdsopen('tt1',-1);
      catch ME
        disp(getReport(ME));
        testCase.assumeFalse(true,...
          sprintf('Could not open tree ''%s'' on ''%s''',tree,server)); % filter test otherwise
      end
      
      val = mdsvalue(node);
      mdsput(node,val+1);
      testCase.assertTrue(mdsvalue(node) == val+1)
      mdsput(node,val); % reset value
      testCase.assertTrue(mdsvalue(node) == val)
    end

    function test_mdstimeout(testCase)
      % Test mdstimeout - disabled should throw error

      mdsconnect(testCase.Server);

      errorid = 'mdsipmex:mdstimeout';

      testCase.verifyError(@() mdstimeout(repmat(1e-3,1,3)),errorid);
    end

    function test_mdsip_connection_error(testCase)
      % Test exception is thrown upon mdsip connection failures

      testCase.assumeTrue('Skipping connection error test')

      mdsconnect(testCase.Server);

      errorid = 'mdsipmex:connection';

      % Break it on purpose (this might not work anymore once we upgrade MDSplus on the server)
      testCase.verifyError(@() mdsvalue('_a=*,_a[0]'),errorid);

      % Check that further calls throw errors as well
      testCase.verifyError(@() mdsvalue('1'),errorid);

    end
    
  end
end
