% Integrates the system with Euler, RK4 and ode45
% X0 : row vector initial condition = [X0,Y0,Z0]
% dt : scalar, time step for integration
% iter : scalar, number of iterations for integration
% XEuler, XRK4 : matrix size (iter+1,3), contain [X(t),Y(t),Z(t)] for iter+1 values of t
% Xode45 : matrix size (n,3), contain [X(t),Y(t),Z(t)] for n values of t chosen by ode45
function [XEuler, XRK4, Xode45] = integration(X0,dt,iter)

% Initialising matrices
XEuler=zeros(iter+1,3); XRK4=zeros(iter+1,3); % Initialise matrices that will contain [X;Y;Z] vectors for all time steps
XEuler(1,:)=X0; XRK4(1,:)=X0; % Setting initial state

for n=1:iter
	t=n*dt; % Time, not useful in our case

    % Euler
    XEuler(n+1,:)= XEuler(n,:)+dt*diffEq(t,XEuler(n,:))';

    % RungeKutta 4
    k1= dt*diffEq(t,XRK4(n,:))';
    k2= dt*diffEq(t+dt/2.,XRK4(n,:)+ k1/2.)';
    k3= dt*diffEq(t+dt/2.,XRK4(n,:)+ k2/2.)';
    k4= dt*diffEq(t+dt,XRK4(n,:)+ k3)';
    XRK4(n+1,:)= XRK4(n,:)+(k1+2*k2+2*k3+k4)/6.;
end

% Matlab ODE
[t,Xode45] = ode45(@diffEq,[0,iter*dt],X0');

end


