% Calculates the correlation dimension for the Lorenz attractor, as in Grassberger and Procaccia 1983

iter=15000; % We try to use the same parameters as in the paper
dt=0.025; % In the paper dt=0.25 is claimed...but it diverges using RK4
        % Largest dt value for which it converges: ~0.1
           % Also, results vary quite a lot with the timestep, with respect
           % to the 0.01 claimed in the paper
N_excl = 0; %points to be excluded from the trajectory (first transient)

% Parameters 
global sigma b r;
sigma=10; 
b=8./3.;
r=40;

X0 = [0,1,0]; % The solution changes slightly with initial conditions
vec=RK4(X0,dt,iter+N_excl);

%% Find the maximum and minimum distance between two points in the trajectory

% We will use the maximum as the length in which all the distances between
% pairs of points are included. The minimum will be instead close to the
% reference l0 distance

max = 0;
min = 1.E99;
eps = 1.e-6;

for i=N_excl+1:iter+N_excl-1 
    for j=i+1:iter+N_excl  % it is not useful to count twice the pairs of points
        sum = 0;
        for k = 1:3
            sum = sum + (vec(i,k) - vec(j,k))^2;
        end
        sum = sqrt(sum);
        if sum > max
            max = sum;
        end
        if sum < min
            min = sum;
        end
    end
end

max = max + eps;
min = min-eps;

%% Counts how many pairs of points are distance apart closer than l for several l values

n_lengths = 14; % number of reference lengths for which the counting is done

n=1;
l = zeros(1,n_lengths);
C = zeros(1,n_lengths);
c_unnorm = zeros(1,n_lengths);
l(1) = 1/(2^(n-1)); % this is the value with respect to the maximum distance
while l(n)*max>min && n<=n_lengths % we stop when l is smaller than the smallest distance 
    count = 0;
    for i=N_excl+1:iter+N_excl-1  
        for j=i+1:iter+N_excl
            sum = 0;
            for k = 1:3
               sum = sum + (vec(i,k) - vec(j,k)).^2;
            end
               sum = sqrt(sum);
            if sum < l(n)*max
                count = count + 1;
            end
        end
    end
    c_unnorm(n) = count; % 
    C(n) = count/((iter^2-iter)/2); % We count only (N^2-N)/2 possible couples of points in a matrix N*N
    n=n+1;
    l(n) = 1/(2.^(n-1));
end

l_p = l(C~=0);
C_p = C(C~=0);
l0 = l_p(end); % Choose the smallest l as the smallest with a positive number of counts

figure
plot(log2(l_p/l0),log2(C_p),'-ko');
axis equal
xlim([0 18])
ylim([-27 2])
xlabel('log_2(l/l_0)')
ylabel('log_2(C)')
title(['r = ',num2str(r)])

p = polyfit(log(l_p(3:end)),log(C_p(3:end)),1); % The first two points are excluded, as in the paper
C_int = exp(p(1)*log(l_p(2:end))+p(2));

hold on
plot(log2(l_p(2:end)/l0),log2(C_int),'--r');

disp(['nu = ',sprintf('%0.5g',p(1))])

hold on
R_box = 4;
Z_box = -24;
text(R_box,Z_box,['\nu = ',sprintf('%0.5g',p(1))]);

% Saves the figure in pdf format
% h = gcf;
% h.PaperPositionMode = 'auto';
% fig_pos = h.PaperPosition;
% fig_pos(2)=fig_pos(2)-15.;
% h.PaperSize = [fig_pos(3)/1.75 fig_pos(4)*1.05];
% % set(gcf,'Units','Inches');
% % pos = get(h,'Position');
% % set(h,'PaperPositionMode','Auto','PaperUnits','Inches','PaperSize',[pos(3), pos(4)-0.5])
% cd ../../ExoFig/
% % print(h,['lor_attr_dim_r',num2str(r)],'-dpdf','-r200','-bestfit')
% print(h,['lor_attr_dim_r',num2str(r)],'-dpdf')
% cd ../ExoNumericalFiles/Lorenz_Attractor_Dimension/
