#!/usr/bin/env python

import ising
import numpy as np
import time
import matplotlib.pylab as plt

# ---------------------------------------------------------------------

ngrid = 16              # we use a 10x10grid 
neq = 100              # Number of equilibration step (multipled by nskip)
nstep= 800             # Number of sampling
nskip = 5              # number of steps to be skipped between two samplings
init_status = 'warm'   # initial status of the spin configuration
kT = 0.7               # temperature
scheme = 'scan'        # flip scheme: either ’random’ or ’scan’
verbose = 1            # enable output


calc  = ising.ising(ngrid, nstep, nskip, neq, init_status, kT, scheme)   # init the ising object
calc.run(verbose)               # self-explanatory

# Show an animation of the spins
calc.animation()

# Plot the magnetization
plt.figure()
plt.plot(calc.arrM)
plt.ylabel('M')
plt.show()



# ---------------------------------------------------------------------
# susceptibility

ngrid = 16
nstep = 100
nskip = ngrid**2
neq = 200
init_status = 'hot'
kT = 2.29
verbose = 0
scheme = 'scan' 
tmin = 1.5
tmax = 3.2
tp = 50            # numbers of temperatures
m = np.zeros((tp,3))
m[:,0] = np.linspace(tmin,tmax,tp)  # list of temperatures to be investigated
nrep = 10 
t_start = time.time()

for i in range(tp):
   print("T = {0:4.2f} ".format(m[i,0]))
   for j in range(nrep):
       print("Trial no.:{0:3}".format(j))
       calc = ising.ising(ngrid, nstep, nskip, neq, init_status, m[i,0], scheme)
       calc.run(verbose)
       print("<M> : {0:8.3f}".format(calc.mavg))
       print("<M2>: {0:8.3f}".format(calc.m2avg))
       m[i,1] += np.abs(calc.mavg)
       m[i,2] += calc.m2avg - calc.mavg**2
   print("Time: {0:6.2f}s".format(time.time()-t_start))
m[:,1] /= nrep            # <M>
m[:,2] /= nrep*kT         # chi  
np.savetxt('ising.dat', m)    

# ---------------------------------------------------------------------
# finite size scaling

nstep = 100
neq = 200
init_status = 'hot'
kT = 2.29
verbose = 0
scheme = 'scan' 
small_grid=20
large_grid=100
nngrid = 6 # Number of grid to consider
nrep = 50
m = np.zeros((nngrid,3))
m[:,0] = np.ceil(np.linspace(small_grid,large_grid,nngrid))
t_start = time.time()
for i in range(nngrid):
   print("L = {0:3} ".format(int(m[i,0])))
   ngrid=int(m[i,0])
   nskip = ngrid**2
   for j in range(nrep):
       print("Trial no.: {0:3}".format(j))
       calc = ising.ising(int(m[i,0]), nstep, nskip, neq, init_status, kT, scheme)
       calc.run(verbose)
       print("<M>: {0:5.3}".format(calc.mavg))
       print("<M2>: {0:5.3}".format(calc.m2avg))
       m[i,1] += np.abs(calc.mavg)
       m[i,2] += calc.m2avg
m[:,1] /= nrep
m[:,2] /= nrep
np.savetxt('finite_size.dat',m)

