#!/usr/bin/python3

# to place 'npart' particles on a simple cubic 
# lattice with density 'rho'


from numba import jit
import math
import random
import numpy as np
import time
import matplotlib
import matplotlib.pyplot as plt

from MC import *

def main():

    # Parameters section begin
    # Simulation parameter
    npart=864       # Number of particles
    rho=0.8072      # Density
    rcut=5.0        # Cutoff radius
    
    # Monte Carlo parameters
    dmax=0.1        # maximal distance in one displacement
    ndispl=50       # number of displacement attempts in one move
    nequil=10000    # number of equilibration cycles
    lmax=2000       # number of production cycles
    fsamp=10        # frequency of sampling (number of sampling lmax/nasmp)
    
    # Parameters of the LJ potential
    eps=1           # L.J. energy          
    sig=1           # L.J. distance 
    #mass=1         # Mass 
    
    # Parameter for s(k)
    qmax= 32        # Max range
    nqvec= 144      # Number of point
    
    # Temperature
    temp=0.7833
    beta=1/temp
    
    nbins=100 # samping for g (r), use even number
    # Parameters section end
    
    
    box=(npart/rho)**(1.0/3)        # length of the cubic lattice
    vol=npart*1.0/rho               # volume of the box
    rcut=min(box/2.0,rcut)          # Check that cutoff is not larger than the box 
    r2cut=rcut**2                   # Squared cutoff raidus
    new_pos=np.array([0.0,0.0,0.0]) # Array for the trial position

    # Set a fixed seed (for debug !!)
    #random.seed(15)
    
    # Energy and pressure correction
    etail= npart*cor_en(rcut, rho, sig, eps)
    ptail= cor_pr(rcut,rho, sig, eps)
    
    # Create initial position
    pos=ini_pos_fcc(npart, box)
    print ("Initialization on a lattice: particles placed on a lattice")
    
    
    a,pr=energy(pos, r2cut, eps, sig, npart, box, rho, vol, beta)
    print ('Initialization: potential energy', a)
    
    ###################################################

    # Timing
    start_time = time.time()
    
    print ('equilibration cycles')
    
    # Search for the optimal maximum displacement (s.t. accepted rate = 0.5)
    frac=0.5
    lfrac5=[]
    for i in range (nequil) :
        naccpt=0
        if math.fmod (i+1,5)==0 :
            # update dmax after every 5 cycles
            dold=dmax
            dmax=dold*(frac/0.5)
            # limit the change in dmax
            if dmax/dold > 1.5 :
              dmax=dold*1.5
            if dmax/dold < 0.5 :
              dmax=dold*0.5
            if dmax > box/2.0 :
              dmax=box/2.0
        
        frac5,_,_=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, False) 
        lfrac5.append(frac5)
        frac=sum(lfrac5)/len(lfrac5)
        if math.fmod (i+1,500)==0 :
          print ('cycle', i+1, 'frac. accp.',round(frac,4),'dmax',round(dmax,4) )

    # Write the position of the atoms in a file
    dump_pos('sample_ar.dat', pos, npart, box, dmax)
    
    ###################################################
    
    print ('production cycles')
    
    # Read the position of the atoms in a file
    pos, npart, box, dmax=read_pos('sample_ar.dat')

    gr=np.zeros(nbins,dtype=int)
    nave=0 # Number of sampling point
    for i in range (lmax) :
        if math.fmod(i+1,fsamp)==0 : # Only compute g(r) and s(k) every fsamp steps
            a1=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, True)
            gofr(gr, pos, box, rho, nbins, npart)
            nave+=1
        else:
            a1=move(pos, new_pos, dmax, ndispl, npart, box, r2cut, eps, sig, beta, rho, vol, False)  
    
    
    # Average g(r)
    gofr_ave=ave_gofr(gr,box,npart,nbins,nave)
    
    
    # Write g(r) in file gr.dat
    g1=open('gr.dat','w')
    g1.write('# r/sigma '+'g (r)'+'\n')
    for i in range (len(gofr_ave['g'])) :
      g1.write( str(gofr_ave['r'][i])+' '+str(gofr_ave['g'][i])+'\n')
    g1.close()
    
    # Write s(k) in file sk.dat
    g2=open('sk.dat','w')
    g2.write('# k*sigma '+'S (k)'+'\n')
    sofk=sofk_FT(gofr_ave, box, npart, qmax, nqvec)
    for i in range(len(sofk['k'])):
        g2.write('{} {}\n'.format(sofk['k'][i],sofk['s'][i]))
    g2.close()
    
    # Plot g(r) and s(k)
    fig1=plt.figure()
    x=gofr_ave['r']
    y=gofr_ave['g']
    plt.plot (x,y)
    plt.xlabel ('r/$\sigma$')
    plt.ylabel ('g (r)')
    plt.savefig ('gofr.pdf')
    
    
    fig2=plt.figure()
    x=sofk['k']
    y=sofk['s']
    plt.plot (x,y)
    plt.xlabel ('k*$\sigma$')
    plt.ylabel ('S (k)')
    plt.savefig ('sofk.pdf')
    
    plt.show()
    
    print("--- %s seconds ---" % (time.time() - start_time))

if __name__=="__main__":
    main()

