%% You have to compare performance and model complexity for variants
%% of SVM (C-SVM and nu-SVM) on 2D Datasets
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%      1) Load 2D Non-Linear Classification Dataset          %%
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 1a) Generate Concentric Circle Data
clear all;
close all;

num_samples = 500;
dim_samples = 2;
num_classes = 2;

[X,labels]  = ml_circles_data(num_samples,dim_samples,num_classes,'svm');
labels      = ml_2binary(labels)';

% Randomize indices of dataset
rand_idx = randperm(num_samples);
X = X(rand_idx,:);
labels = labels(rand_idx);

% Plot data
plot_options            = [];
plot_options.is_eig     = false;
plot_options.labels     = labels;
plot_options.title      = 'Concentric Circles Dataset';

if exist('h1','var') && isvalid(h1), delete(h1);end
h1 = ml_plot_data(X,plot_options);
legend('Class 1', 'Class -1')
axis equal

%% 1b) Generate Checkerboard Data
clear all;
close all;

num_samples_p_quad = 100; % Number of points per quadrant
[X,labels] = ml_checkerboard_data(num_samples_p_quad);

% Plot data
plot_options            = [];
plot_options.is_eig     = false;
plot_options.labels     = labels;
plot_options.title      = 'Checkerboard Dataset';

if exist('h1','var') && isvalid(h1), delete(h1);end
h1 = ml_plot_data(X,plot_options);
legend('Class 1', 'Class -1')
axis equal

%% 1c) Load Very Non-linear data

clear all;
close all;
load('very-nonlinear-data')

% Plot data
plot_options            = [];
plot_options.is_eig     = false;
plot_options.labels     = labels;
plot_options.title      = 'Very Non-Linear Dataset';

if exist('h1','var') && isvalid(h1), delete(h1);end
h1 = ml_plot_data(X,plot_options);
legend('Class 1', 'Class -1')
axis equal

%% 1d) Load Ripley Dataset

clear all;
close all;

load('rvm-ripley-dataset')
labels      = ml_2binary(labels)';
% Plot data
plot_options            = [];
plot_options.is_eig     = false;
plot_options.labels     = labels;
plot_options.title      = 'Ripley Dataset';

if exist('h1','var') && isvalid(h1), delete(h1);end
h1 = ml_plot_data(X,plot_options);
legend('Class 1', 'Class -1')
axis equal

%% 1e) Draw Data with ML_toolbox GUI
clear all; close all; clc;
[X,labels] = ml_draw_data;
[N,M] = size(X);
X       = X';
labels  = ml_2binary(labels)';

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%               2) Learn C-SUPPORT VECTOR MACHINE              %%
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 2a) EXAMPLE SVM OPTIONS FOR C-SVM with RBF Kernel%
clear options
options.svm_type    = 0;    % 0: C-SVM, 1: nu-SVM
options.kernel_type = 0;    % 0: RBF, 1: Polynomial (0 is the default)
options.C           = 1000;    % Cost (C \in [1,1000]) in C-SVM
options.sigma       = 5;   % radial basis function: exp(-gamma*|u-v|^2), gamma = 1/(2*sigma^2)

%% 2b) EXAMPLE SVM OPTIONS FOR C-SVM with Polynomial Kernel% 
clear options
options.svm_type    = 0;    % 0: C-SVM, 1: nu-SVM
options.kernel_type = 1;    % 0: RBF, 1: Polynomial
options.C           = 10;  % Cost (C \in [1,1000]) in C-SVM
options.degree      = 2;    % polynomial kernel: (<x,x^i> + coeff)^degree
options.coeff       = 0;    % polynomial kernel: (<x,x^i> + coeff)^degree, when coeff=0 homogeneous, coeff>=1 inhomogeneous

%% Train SVM Classifier
[predicted_labels, model] = svm_classifier(X, labels, options, []);

% Plot SVM decision boundary
ml_plot_svm_boundary( X, labels, model, options, 'draw');

%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Do grid-search to find 'optimal' hyper-parameters for C-SVM with RBF
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 2c)  Set options for SVM Grid Search and Execute
clear options
options.svm_type   = 0;         % SVM Type (0:C-SVM, 1:nu-SVM)
options.limits_C   = [1, 10000];  % Limits of penalty C
options.limits_w   = [0.1, 2]; % Limits of kernel width \sigma
options.steps      = 10;        % Step of parameter grid 
options.K          = 10;        % K-fold CV parameter
options.log_grid   = 1;         % Create log-sampled grid, set 0 for linear

% Do Grid Search
[ ctest, ctrain , cranges ] = ml_grid_search_class( X, labels, options );

% Extract parameter ranges
range_C  = cranges(1,:);
range_w  = cranges(2,:);

% Get CV statistics
stats = ml_get_cv_grid_states(ctest,ctrain);

% Visualize Grid-Search Heatmap
cv_plot_options              = [];
cv_plot_options.title        = strcat('C-SVM :: ', num2str(options.K),'-fold CV with RBF');
cv_plot_options.param_names  = {'C', '\sigma'};
cv_plot_options.param_ranges = [range_C ; range_w];
cv_plot_options.log_grid     = options.log_grid ;      

if exist('hcv','var') && isvalid(hcv), delete(hcv);end
hcv = ml_plot_cv_grid_states(stats,cv_plot_options);

% Find 'optimal hyper-parameters'
[max_acc,ind] = max(stats.test.acc.mean(:));
[C_max, w_max] = ind2sub(size(stats.test.acc.mean),ind);
C_opt     = range_C(C_max)
sigma_opt = range_w(w_max)

%% 2d) Plot Decision Boundary with 'Optimal' Hyper-parameters
options.svm_type = 0;    
options.C        = round(C_opt);
options.sigma    = sigma_opt; 

% OR

% Manually Choose other hyper-parameters form visually inspecting Heatmap
% options.C        = 166;
% options.sigma    = 0.73; 

% Train SVM Classifier
[predict_labels, model] = svm_classifier(X, labels, options, []);

% Plot SVM decision boundary
ml_plot_svm_boundary( X, labels, model, options, 'draw');


%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Do grid-search to find 'optimal' hyper-parameters for C-SVM with Poly
% Kernel
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 2e) Do Grid Search on SVM with polynomial kernels
clear options
options.svm_type    = 0;         % SVM Type (0:C-SVM, 1:nu-SVM)
options.kernel_type = 1;         % 0: RBF, 1: Polynomial
options.limits_C    = [1, 100];  % Limits of penalty C
options.limits_d    = [1, 5];    % Limits of polynomial degree
options.coeff       = 1;         % 0:, Homogeneous Polynomial, 1: Inhomogeneous Polynomial,
options.steps       = 5;         % Step of parameter grid 
options.K           = 10;        % K-fold CV parameter
options.log_grid    = 0;         % Create log-sampled grid, set 0 for linear

% Do Grid Search
[ ctest, ctrain , cranges ] = ml_grid_search_class( X, labels, options );

% Extract parameter ranges
range_C  = cranges(1,:);
range_d  = cranges(2,:);

% Get CV statistics
stats = ml_get_cv_grid_states(ctest,ctrain);

% Visualize Grid-Search Heatmap
cv_plot_options              = [];
cv_plot_options.title        = strcat('C-SVM :: ', num2str(options.K),'-fold CV Inhomogeneous Poly');
cv_plot_options.param_names  = {'C', 'degree'};
cv_plot_options.param_ranges = [range_C ; range_d];
cv_plot_options.log_grid     = options.log_grid ;   

if exist('hcv','var') && isvalid(hcv), delete(hcv);end
hcv = ml_plot_cv_grid_states(stats,cv_plot_options);

% Find 'optimal hyper-parameters'
[max_acc,ind] = max(stats.test.acc.mean(:));
[C_max, d_max] = ind2sub(size(stats.test.acc.mean),ind);
C_opt = range_C(C_max);
d_opt = range_d(d_max);


%% 2f) Plot Decision Boundary with 'Optimal' Hyper-parameters
clear options
options.svm_type    =  0;    
options.kernel_type = 1;    
options.C           = C_opt;
options.degree      = d_opt;  
options.coeff       = 1;

% Train SVM Classifier
[predict_labels, model] = svm_classifier(X, labels, options, []);

% Plot SVM decision boundary
ml_plot_svm_boundary( X, labels, model, options, 'draw');


%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%               3) Learn nu-SUPPORT VECTOR MACHINE             %%
%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 3a) EXAMPLE SVM OPTIONS FOR nu-SVM with RBF Kernel% 
clear options
options.svm_type    = 1;    % 0: C-SVM, 1: nu-SVM
options.nu          = 0.05; % nu \in (0,1) (upper-bound for misclassifications on margni and lower-bound for # of SV) for nu-SVM
options.sigma       = 10;    % radial basis function: exp(-gamma*|u-v|^2), gamma = 1/(2*sigma^2)

%% 3b) EXAMPLE SVM OPTIONS FOR nu-SVM with Polynomial Kernel% 
clear options
options.svm_type    = 1;    % 0: C-SVM, 1: nu-SVM
options.kernel_type = 1;    % 0: RBF, 1: Polynomial
options.nu          = 0.2;  % nu \in (0,1) (upper-bound for misclassifications on margin and lower-bound for # of SV) for nu-SVM
options.degree      = 4;    % polynomial kernel: (<x,x^i> + coeff)^degree
options.coeff       = 0;    % polynomial kernel: (<x,x^i> + coeff)^degree, when coeff=0 homogeneous, coeff>=1 inhomogeneous

%% Train SVM Classifier
[predicted_labels, model] = svm_classifier(X, labels, options, []);

% Plot SVM decision boundary
ml_plot_svm_boundary( X, labels, model, options, 'surf');


%% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Do grid-search to find 'optimal' hyper-parameters for nu-SVM with RBF
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 3c)  Set options for nu-SVM Grid Search and Execute
clear options
options.svm_type   = 1;           % SVM Type (0:C-SVM, 1:nu-SVM)
options.limits_nu  = [0.01, 0.5]; % Limits of penalty nu
options.limits_w   = [0.05, 1];  % Limits of kernel width \sigma
options.steps      = 10;          % Step of parameter grid 
options.K          = 10;          % K-fold CV parameter
options.log_grid   = 1;           % Create log-sampled grid, set 0 for linear

% Do Grid Search
[ ctest, ctrain , cranges ] = ml_grid_search_class( X, labels, options );

% Extract parameter ranges
nu_range = cranges(1,:);
w_range  = cranges(2,:);

% Get CV statistics 
stats = ml_get_cv_grid_states(ctest,ctrain);

% Visualize Grid-Search Heatmap
cv_plot_options              = [];
cv_plot_options.title        = strcat('\nu-SVM :: ', num2str(options.K),'-fold CV with RBF');
cv_plot_options.param_names  = {'\nu', '\sigma'};
cv_plot_options.param_ranges = [nu_range ; w_range];
cv_plot_options.log_grid     = options.log_grid ;  

if exist('hcv','var') && isvalid(hcv), delete(hcv);end
hcv = ml_plot_cv_grid_states(stats,cv_plot_options);

% Find 'optimal hyper-parameters'
[max_acc,ind]   = max(stats.test.acc.mean(:));
[nu_max, w_max] = ind2sub(size(stats.test.acc.mean),ind);
nu_opt = nu_range(nu_max)
w_opt  = w_range(w_max)

%% 3d) Plot Decision Boundary with 'Optimal' Hyper-parameters
clear options
options.svm_type = 1;    
options.nu       = nu_opt;
options.sigma    = w_opt; 

% OR

% Manually Choose other hyper-parameters form visually inspecting Heatmap
% options.nu       = 0.13;
% options.sigma    = 0.20; 

% Train SVM Classifier
[predict_labels, model] = svm_classifier(X, labels, options, []);

% Plot SVM decision boundary
ml_plot_svm_boundary( X, labels, model, options, 'draw');
