import sys
from PyQt5 import QtWidgets, QtCore, uic

from MainWindow import Ui_MainWindow
from time import sleep

import serial.tools.list_ports
from pySerialTransfer import pySerialTransfer as txfer

from datetime import datetime
import pandas as pd
import os

from pyqtgraph import PlotWidget, plot
import pyqtgraph as pg
import matplotlib.pyplot as plt
import numpy as np
from collections import defaultdict

class tx_command(object):
    key = 'a'
    value_1 = 32.0
    value_2 = 32.0


class MainWindow(QtWidgets.QMainWindow, Ui_MainWindow):
    def __init__(self, *args, obj=None, **kwargs):
        super(MainWindow, self).__init__(*args, **kwargs)
        self.setupUi(self)

        # TO ADD to in .ui file
    

        # define variables
        # for sending command
        self.new_command = tx_command

        # for saving
        self.savingData = False
        self.recordedDf = pd.DataFrame(columns = ['ts', 'Setpoint1', 'Obj1', 'ATS1_R', 'TEC1_val', 'TEC1_V', 'TEC1_I'])
        # for plotting
        self.plottingData = False
        self.x = list(range(500)) # 50 Hz x 10 seconds = 500 x-points
        # y-data initializion
        self.y_Temp = 32*np.ones((3,500))
        self.keys_Temp = ['ATS1', 'Setpt1', 'Obj1']
        self.y_Out = 2*np.ones((2,500)) # only V and I
        self.keys_Out = ['V_1','I_1']


        # define the callbacks
        self.verticalSlider_1.valueChanged.connect(self.slidervaluechange)
        self.doubleSpinBox_1.valueChanged.connect(self.spinvaluechange)
        self.comboBox.activated.connect(self.statechange)
        self.scanButton.clicked.connect(self.scanclicked)
        self.connectButton.clicked.connect(self.connectclicked)
        self.disconnectButton.clicked.connect(self.disconnectclicked)
        self.plotButton.clicked.connect(self.plotclicked)
        self.saveButton.clicked.connect(self.saveclicked)
        self.pushButtonsetPWM.clicked.connect(self.setPWM)
        self.pushButtonsetATS1_T0.clicked.connect(self.set_ATS1_T0)
        self.pushButtonsetATS1_T1.clicked.connect(self.set_ATS1_T1)
        self.pushButtonsetATS1_R0.clicked.connect(self.set_ATS1_R0)
        self.pushButtonsetATS1_R1.clicked.connect(self.set_ATS1_R1)

        #define the timers
        self.timer_serialread = QtCore.QTimer()
        self.timer_serialread.setInterval(10) # ms
        self.timer_serialread.timeout.connect(self.check_incoming_data)

        self.timer_updateplots = QtCore.QTimer()
        self.timer_updateplots.setInterval(100) # ms
        self.timer_updateplots.timeout.connect(self.update_plots)

        self.timer_sendData = QtCore.QTimer()
        self.timer_sendData.setInterval(100) # ms
        self.timer_sendData.timeout.connect(self.sendData)

        self.timer_sendDirectCtrl = QtCore.QTimer()
        self.timer_sendDirectCtrl.setInterval(100) # ms
        self.timer_sendDirectCtrl.timeout.connect(self.sendDirectCtrl)

    def set_ATS1_R0(self):
        self.doubleSpinBox_ATS1_R0.setValue(self.ATS1_R)

    def set_ATS1_T0(self):
        self.new_command.key = 'c'
        self.new_command.value_1 =  float(self.doubleSpinBox_ATS1_R0.value())
        self.new_command.value_2 =  float(self.doubleSpinBox_ATS1_T0.value())
        self.sendData()

    def set_ATS1_R1(self):
        self.doubleSpinBox_ATS1_R1.setValue(self.ATS1_R)

    def set_ATS1_T1(self):
        self.new_command.key = 'C'
        self.new_command.value_1 =  float(self.doubleSpinBox_ATS1_R1.value())
        self.new_command.value_2 =  float(self.doubleSpinBox_ATS1_T1.value())
        self.sendData()

    def setPWM(self):
        self.new_command.key = 'p'
        self.new_command.value_1 =  float(self.doubleSpinBox_heater1.value()/100.)# value in % to translate as a float between 0 and 1
        self.new_command.value_2 =  0.0
        self.sendData()

    def sendDirectCtrl(self):
        # send the new value to the MT
        self.new_command.value_1 =  3110.0-float(self.verticalSlider_dc_1.value())# inversed axis because it is connected in reverse
        self.new_command.value_2 =  1555.0
        self.sendData()

    def sendData(self):
        sendSize = 0
        #print(float(self.verticalSlider_1.value())/10)
        sendSize = self.link.tx_obj(self.new_command.key, start_pos = sendSize)
        sendSize = self.link.tx_obj(self.new_command.value_1, start_pos = sendSize)
        sendSize = self.link.tx_obj(self.new_command.value_2, start_pos = sendSize)
        self.link.send(sendSize)
        

    def update_plots(self):

        for i in range(3):# Temperature plot
            self.lines_Temp[i].set_ydata(self.y_Temp[i])
            self.texts_Temp[i].set_text("{}: {:.1f}".format(self.keys_Temp[i], np.mean(self.y_Temp[i][450:]))) # Update the text with the current temperature

        for i in range(2):# Output plot
            self.lines_Out[i].set_ydata(self.y_Out[i])
            self.texts_Out[i].set_text("{}: {:.1f}".format(self.keys_Out[i], np.mean(self.y_Out[i][450:]))) # Update the text with the current temperature

        self.FigPlotTemp.canvas.draw()
        self.FigPlotOut.canvas.draw()
 

    def check_incoming_data(self):
        # check for incoming data
        # if there is
        #     print the data in the textEdit
        if self.link.available():
            recSize = 0

            ts = self.link.rx_obj(obj_type='i', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['i']
            # module 1
            Setpoint1 = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            Obj1 = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            ATS1_R = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            ATS1_T = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            TEC1_val = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            TEC1_V = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']
            TEC1_I = self.link.rx_obj(obj_type='f', start_pos=recSize)
            recSize += txfer.STRUCT_FORMAT_LENGTHS['f']


            self.textEdit.setPlainText('ts:{} \t ATS_R: {:.2f} \tObject: {:.2f} \t TEC_val: {:.2f} \n' .format(ts,ATS1_R, Obj1, TEC1_val))

            # keep for calibration
            self.ATS1_R = ATS1_R

            
            if self.savingData:
                dfnew = pd.DataFrame({
                    'ts' : [ts],
                    'Setpoint1': [Setpoint1],
                    'Obj1': [Obj1], 
                    'ATS1_R': [ATS1_R],
                    'ATS1_T': [ATS1_T], 
                    'TEC1_val': [TEC1_val],  
                    'TEC1_V': [TEC1_V], 
                    'TEC1_I': [TEC1_I], 
                })
                self.recordedDf = pd.concat([self.recordedDf, dfnew], ignore_index=True)

            if self.plottingData:
                self.y_Temp = np.roll(self.y_Temp, -1, axis=1)# shift the array by 1 element 
                # replace the last element (oldest becasue of the roll) by new data
                self.y_Temp[:,-1]= np.transpose([ATS1_T, Setpoint1, Obj1])

                # same for output data
                self.y_Out = np.roll(self.y_Out, -1, axis=1)
                self.y_Out[:,-1]= np.transpose([TEC1_V, TEC1_I])


        elif self.link.status < 0:
            if self.link.status == txfer.CRC_ERROR:
                print('ERROR: CRC_ERROR')
            elif self.link.status == txfer.PAYLOAD_ERROR:
                print('ERROR: PAYLOAD_ERROR')
            elif self.link.status == txfer.STOP_BYTE_ERROR:
                print('ERROR: STOP_BYTE_ERROR')
            else:
                print('ERROR: {}' .format(self.link.status))

    def plotclicked(self):
        self.plottingData = True

        # initialize the plots
        # plot for the temperatures
        self.FigPlotTemp = plt.figure(figsize=(10,4))
        self.AxPlotTemp = self.FigPlotTemp.add_subplot(1,1,1)
        plt.subplots_adjust(left=0.06, bottom=0.15, right=0.9)
        self.AxPlotTemp.set_xlim(left = 0, right = 500)
        self.AxPlotTemp.set_ylim(bottom = 12, top = 45)
        self.AxPlotTemp.set_ylabel('temperature [°C]')
        self.AxPlotTemp.set_xlabel('samples')
        self.AxPlotTemp.set_title('Temperatures Stream')
        self.AxPlotTemp.spines['right'].set_visible(False)
        self.AxPlotTemp.spines['top'].set_visible(False)
        self.AxPlotTemp.xaxis.set_ticks_position('bottom')
        self.AxPlotTemp.yaxis.set_ticks_position('left')
        
        # initialize the lines and text
        self.lines_Temp = defaultdict(lambda: defaultdict(dict))
        self.texts_Temp = defaultdict(lambda: defaultdict(dict))
        y_text = 0.9
        linesstyles = [':', ':', '-.']
        linescolors = ['silver', 'black', 'black']
        self.AxPlotTemp.text(1.01, 0.98, '1[s] mean:', transform=self.AxPlotTemp.transAxes, fontweight='bold')
        for i in range(3):
            line, = self.AxPlotTemp.plot(self.x, self.y_Temp[i], color=linescolors[i], linestyle=linesstyles[i], label=self.keys_Temp[i])
            self.lines_Temp[i] = line
            self.texts_Temp[i] = self.AxPlotTemp.text(1.01, y_text, '', transform=self.AxPlotTemp.transAxes) # Create a blank text to be updated in animate
            y_text -= 0.06
        
        self.AxPlotTemp.legend(loc='upper left', ncol=1, bbox_to_anchor=(1, 0.5))
        plt.show() # make the plot visible

        # plot for the outputs
        self.FigPlotOut = plt.figure(figsize=(10,4))
        self.AxPlotOut = self.FigPlotOut.add_subplot(1,1,1)
        plt.subplots_adjust(left=0.06, bottom=0.15, right=0.9)
        self.AxPlotOut.set_xlim(left = 0, right = 500)
        self.AxPlotOut.set_ylim(bottom = -5, top = 5)
        self.AxPlotOut.set_ylabel('Output')
        self.AxPlotOut.set_xlabel('samples')
        self.AxPlotOut.set_title('Outputs Stream')
        self.AxPlotOut.spines['right'].set_visible(False)
        self.AxPlotOut.spines['top'].set_visible(False)
        self.AxPlotOut.xaxis.set_ticks_position('bottom')
        self.AxPlotOut.yaxis.set_ticks_position('left')
        
        # initialize the lines and text
        self.lines_Out = defaultdict(lambda: defaultdict(dict))
        self.texts_Out = defaultdict(lambda: defaultdict(dict))
        y_text = 0.9
        linesstyles = ['-', '-']
        linescolors = ['black', 'silver']
        self.AxPlotOut.text(1.01, 0.98, '1[s] mean:', transform=self.AxPlotOut.transAxes, fontweight='bold')
        for i in range(2):
            line, = self.AxPlotOut.plot(self.x, self.y_Temp[i], color=linescolors[i], linestyle=linesstyles[i], label=self.keys_Out[i])
            self.lines_Out[i] = line
            self.texts_Out[i] = self.AxPlotOut.text(1.01, y_text, '', transform=self.AxPlotOut.transAxes) # Create a blank text to be updated in animate
            y_text -= 0.06
        
        self.AxPlotOut.legend(loc='upper left', ncol=1, bbox_to_anchor=(1, 0.5))


        plt.show() # make the plot visible

        self.timer_updateplots.start() 
        

    def saveclicked(self):
        if self.savingData:
            self.saveButton.setText("record")
            # here save the data to .csv
            self.savingData = False
            self.savinglabel.setText("")

            # save to .csv file

            #dir = os.path.dirname(os.path.realpath(__file__)) + '/logs/' # in case of python file
            dir = os.path.dirname(os.path.realpath(sys.executable)) + '/logs/' # in case of executable file
            if not os.path.exists(dir): # check if the folder exists
                os.makedirs(dir)

            timestring = datetime.now().strftime("%Y%m%d_%H_%M_%S_%f")# YearMonthDay_hours_minutes_seconds_microseconds_data.csv
            self.recordedDf.to_csv(dir + timestring + '_data.csv')
            print('Data saved to ' + timestring + '_data.csv')
            self.recordedDf.drop(self.recordedDf.index, inplace=True)
            #print(self.recordedData.ts)
        else:
            self.saveButton.setText("save to .csv")
            self.savinglabel.setText("recording...")
            self.savingData = True
        

    def scanclicked(self):

        self.comboBox_port.clear()
        #scan the serial ports
        self.ports = serial.tools.list_ports.comports()
        # update comboBox
        for p in self.ports:
            self.comboBox_port.addItem(p.device)

        # dis/able connect button
        if self.comboBox_port.count():
            self.connectButton.setEnabled(True)
            self.textEdit.setPlainText('{} port(s) found' .format(self.comboBox_port.count()))
        else:
            self.connectButton.setEnabled(False)
            self.textEdit.setPlainText("No port found")

    def connectclicked(self):
        # try to connect to the comboBox port at the comboBox_baudrate value
        # if connects
        #     start timer_serialread
        #     able the disconnect button
        #     disable comboBox_baudrate
        #     disable the connect button
        #     disable the comboBox
        #     print message
        # else
        #    print message

        try:
            self.link = txfer.SerialTransfer(port = self.comboBox_port.currentText(), 
                                                baud = int(self.comboBox_baudrate.currentText()))
            self.link.open()
            sleep(1) # allow some time for the Arduino to completely reset
            print("Port opened")
            self.disconnectButton.setEnabled(True)
            self.plotButton.setEnabled(True)
            self.saveButton.setEnabled(True)
            self.timer_serialread.start()
            self.comboBox.setEnabled(True)
        except:
            print("Could not open the port")

       
    def disconnectclicked(self):
        # disconnect to the serial port
        # able t
        # print message
        self.link.close()
        print("Port closed")
        self.disconnectButton.setEnabled(False)
        self.plotButton.setEnabled(False)
        self.saveButton.setEnabled(False)
        self.timer_serialread.stop()
        self.timer_sendData.stop()
        self.comboBox.setEnabled(False)

    def slidervaluechange(self):
        newTemp = self.verticalSlider_1.value()/10 #values of the verticalSlider_1 are integer between 150 and 420
        self.doubleSpinBox_1.setValue(newTemp)
        # send the new value to the MT
        self.new_command.value_1 =  float(self.verticalSlider_1.value())/10
        self.new_command.value_2 =  0/10
        self.sendData() 

    def spinvaluechange(self):
        newTemp = self.doubleSpinBox_1.value()
        self.verticalSlider_1.setValue(int(newTemp*10))
        # send the new value to the MT
        self.new_command.value_1 =  float(self.verticalSlider_1.value())/10
        self.new_command.value_2 =  float(0)/10
        self.sendData() 

    def statechange(self):
        current_index = self.comboBox.currentIndex()
        if current_index == 0:
            print("IDLE")
            self.verticalSlider_1.setEnabled(False)
            self.doubleSpinBox_1.setEnabled(True)
            self.verticalSlider_dc_1.setEnabled(False)
            # send the command that we are in ATS tracking mode
            self.new_command.key = 'i'
            self.new_command.value_1 = 0.0
            self.new_command.value_2 = 0.0
            self.sendData() 
            self.timer_sendDirectCtrl.stop()
        elif current_index == 1:
            print("TRACKING ATS")
            self.verticalSlider_1.setEnabled(False)
            self.doubleSpinBox_1.setEnabled(False)
            self.verticalSlider_dc_1.setEnabled(False)
            # send the command that we are in ATS tracking mode
            self.new_command.key = 'a'
            self.new_command.value_1 = 0.0
            self.new_command.value_2 = 0.0
            self.sendData() 
            #self.timer_sendData.start()
            self.timer_sendDirectCtrl.stop()
        elif current_index == 2:
            print("TRACKING GUI")
            self.verticalSlider_1.setEnabled(True)
            self.doubleSpinBox_1.setEnabled(True)
            self.verticalSlider_dc_1.setEnabled(False)
            self.new_command.key = 'g'
            self.new_command.value_1 = float(self.verticalSlider_1.value())/10
            self.new_command.value_2 = float(0)/10
            self.sendData()
            #self.timer_sendData.start()
            self.timer_sendDirectCtrl.stop()
        else:
            print("DIRECT CONTROL")
            self.verticalSlider_1.setEnabled(False)
            self.doubleSpinBox_1.setEnabled(False)
            self.verticalSlider_dc_1.setEnabled(True)
            self.new_command.key = 'd'
            self.new_command.value_1 = float(self.verticalSlider_dc_1.value())
            self.new_command.value_2 = float(0)
            self.sendData()
            self.timer_sendDirectCtrl.start()


app = QtWidgets.QApplication(sys.argv)
app.setStyle('Fusion')
window = MainWindow()
window.show()
app.exec()

