#!/usr/bin/env python
import asyncio
import threading
import os
import signal
import tempfile
import json
from datetime import datetime

import tkinter as tk
from tkinter import ttk
import sv_ttk

from collections import deque

from ControllerWithGUIMicroDelta.gui.gui_microDelta import gui
from ControllerWithGUIMicroDelta.MicroDelta.delta_robot import delta_robot
from ControllerWithGUIMicroDelta.settings import *


async def update(delta: delta_robot,gui_main: gui):
    deque_robot_state = deque(maxlen=int(round(SNAPSHOT_TIME_RANGE / REFRESH_RATE)))
    # maxlen parameter is used to limit the size of the deque, adding new elements will remove the oldest ones

    #await asyncio.sleep(5)
    gui_main.messageDelta.configure(text="Launch update")

    while True:
        # Robot Delta
        _,delta.position[0],delta.position[1],delta.position[2] = delta.kinematics.forward(delta.theta[0],delta.theta[1],delta.theta[2])
    
        # GUI -> Robot Delta
        if (gui_main.robotState != "none"):
            delta.state = gui_main.robotState
            gui_main.robotState = "none"

        delta.target = [gui_main.xTarget, gui_main.yTarget, gui_main.zTarget]
        delta.newTarget = gui_main.newTarget
        if delta.newTarget:
            gui_main.newTarget = False

        delta.PID['kp'] = gui_main.PID['kp']
        delta.PID['ki'] = gui_main.PID['ki']
        delta.PID['kd'] = gui_main.PID['kd']

        delta.newParams = gui_main.newParams
        if delta.newParams:
            gui_main.newParams = False
        
        delta.vmax = gui_main.max_speed
        delta.amax = gui_main.max_acc
        delta.dmax = gui_main.max_dec

        delta.motorControlID = int(gui_main.choiceAngle.get())
        delta.targetAngle = gui_main.targetAngle
        delta.newTargetAngle = gui_main.newTargetAngle
        delta.trajectoryChoice = int(gui_main.trajectoryChoice.get())
        if delta.newTargetAngle:
            gui_main.newTargetAngle = False

        # Robot Delta -> Gui
        for i in range(3):
            # Update value of Theta
            gui_main.thetaValue[i].configure(text='{:.2f}'.format(delta.theta[i]))
            # Update value of Theta for the plot
            gui_main.motors_gui.value_plot.motors_value[i].position = delta.theta[i]

        # Update value XYZ
        gui_main.xValue.configure(text='{:.2f}'.format(delta.position[0]))
        gui_main.yValue.configure(text='{:.2f}'.format(delta.position[1]))
        gui_main.zValue.configure(text='{:.2f}'.format(delta.position[2]))
        
        # Update value XYZ for the plot
        gui_main.motors_gui.value_plot.x = delta.position[0]
        gui_main.motors_gui.value_plot.y = delta.position[1]
        gui_main.motors_gui.value_plot.z = delta.position[2]

        # Update value for the snapshot
        current_time = asyncio.get_event_loop().time()
        state_data = [current_time]
        for i in range(3): # Add all motor angles
            state_data.append(round(delta.theta[i], 5))
        for i in range(3): # Add all position coordinates
            state_data.append(round(delta.position[i], 5))
        deque_robot_state.append(tuple(state_data))

        # Update state of the robot
        gui_main.stateLabel.configure(text=delta.state)
        if delta.message2print:
            gui_main.messageDelta.configure(text=delta.message)
            delta.message2print =False
        
        # Take snapshot 
        if gui_main.want_snapshot:
            gui_main.want_snapshot = False
            gui_main.messageGui.configure(text="Taking snapshot...")
            folder_path = gui_main.current_snapshot_path.get()

            take_snapshot(folder_path, deque_robot_state)
            gui_main.messageGui.configure(text="Snapshot saved")

        # Block GUI
        block_frame(delta.state, "Position", gui_main.targetDeltaFrame)
        block_frame(delta.state, "Angle", gui_main.targetMotorFrame)
        block_frame(delta.state, "Params", gui_main.pidParamsFrame)
        block_frame(delta.state, "Params", gui_main.trajectoryParamsFrame)

        if not delta.serial_channel_loaded_and_valid:
            gui_main.messageDelta.configure(text="Serial channel not loaded or invalid")

        await asyncio.sleep(REFRESH_RATE)

def take_snapshot(folder_path, deque_robot_state):
    # Save the current plot as a PNG file in the selected folder
    timestamp = datetime.now().strftime("%Y-%m-%d-%H%M%S")  # Format: YYYY-MM-dd-hhmmss
    snapshot_path = f"{folder_path}/snapshot_{timestamp}.csv"
    
    # Save the current plot figure
    with open(snapshot_path, "w") as log_file:
        #log_file.write(f"{asyncio.get_event_loop().time()},{delta.position[0]}\n")
        log_file.write("Time,Theta1,Theta2,Theta3,X,Y,Z\n")
        for state in deque_robot_state:
            log_file.write(','.join(map(str, state)) + '\n')

def block_frame(robotstate, state, frame):
    if robotstate == state:
        for child in frame.winfo_children():
            if isinstance(child, ttk.Button):
                pass
            else:
                child.configure(state='normal')
    else:
        for child in frame.winfo_children():
            child.configure(state='disable')


def _asyncio_thread(async_loop, delta: delta_robot, gui_main: gui):
    async_loop.create_task(delta.can_comunication())
    async_loop.create_task(update(delta, gui_main))
    async_loop.run_forever()

def load_path_com_settings():
    com_confirmed = False

    temp_dir = tempfile.gettempdir()
    temp_file_path = os.path.join(temp_dir, TEMP_FILE_PARAM)

    if os.path.exists(temp_file_path):
        with open(temp_file_path, 'r') as file:
            settings = json.load(file)
            serial_channel_loaded = settings.get('SERIAL_CHANNEL', '/dev/ttyACM0')
    else:
        serial_channel_loaded = SERIAL_CHANNEL + ""
        print("Defaulting value to SERIAL_CHANNEL: ", serial_channel_loaded)


    # Create a minimalist GUI
    root = tk.Tk()
    root.geometry("300x150")
    root.title("uDelta COM")

    # Label
    label = ttk.Label(root, text="uDelta COM port")
    label.pack(pady=10)

    # Text entry
    com_entry = ttk.Entry(root)
    com_entry.insert(0, serial_channel_loaded)
    com_entry.pack(pady=5)

    # Button
    def load_com():
        nonlocal serial_channel_loaded, com_confirmed
        serial_channel_loaded = com_entry.get()
        root.destroy()

        # Save the serial channel to a temporary file
        temp_dir = tempfile.gettempdir()
        temp_file_path = os.path.join(temp_dir, TEMP_FILE_PARAM)
        settings = {'SERIAL_CHANNEL': serial_channel_loaded}
        with open(temp_file_path, 'w') as file:
            json.dump(settings, file)
        print("Serial channel saved to temporary file:", temp_file_path)

        com_confirmed = True

    ok_button = ttk.Button(root, text="OK, load COM", command=load_com)
    ok_button.pack(pady=10)

    sv_ttk.set_theme("light")
    root.mainloop()

    return com_confirmed, serial_channel_loaded

def main():
    com_confirmed, serial_channel_loaded = load_path_com_settings()
    if not com_confirmed:
        return

    async_loop = asyncio.new_event_loop()
    gui_main = gui()
    delta = delta_robot(serial_channel_loaded)
    
    thread = threading.Thread(target=_asyncio_thread, args=(async_loop, delta, gui_main))
    thread.start()
    
    gui_main.start_gui()

    thread.join(0.2)
    if thread.is_alive():
        async_loop.call_soon_threadsafe(async_loop.stop)

    os.kill(os.getpid(), signal.SIGTERM)


    
if __name__ == "__main__":
    main()

