import math

## Trajectory Calculator
# This function calculates the length and the direction vector of a trajectory.
def calculate_trajectory(x_start, y_start, z_start, x_end, y_end, z_end):
    # Calculate the length of the trajectory
    traj_length = math.sqrt((x_end - x_start)**2 + (y_end - y_start)**2 + (z_end - z_start)**2)

    # Calculate the direction vector
    vect_dir = [(x_end - x_start)/traj_length, (y_end - y_start)/traj_length, (z_end - z_start)/traj_length]

    return traj_length, vect_dir

## Parabolic Profile Solver
# This function calculates the parameters of a parabolic profile for a given trajectory length and maximum acceleration and deceleration.
def solve_parabolic_profile(traj_length, G_mx, G_mxd, V_init, V_fin, V_mx, Tj, t_suppl):
    COEF_PROFILE_PARABOLIC = 2.0  # replace with actual value

    profile_settings = {
        'acc_max': G_mx,
        'dec_max': G_mxd,
        'v_max': V_mx,
        't_zero': t_suppl
    }

    acc_equiv = COEF_PROFILE_PARABOLIC * ((G_mx * G_mxd) / (G_mx + G_mxd))
    max_length = (V_mx * V_mx) / acc_equiv

    if traj_length <= max_length:
        profile_settings['t_end'] = (4 * traj_length / acc_equiv) ** 0.5
        profile_settings['t_acc'] = (G_mxd * profile_settings['t_end']) / (G_mx + G_mxd)
        profile_settings['t_dec'] = profile_settings['t_acc']
        profile_settings['v_max'] = (G_mx * COEF_PROFILE_PARABOLIC * profile_settings['t_acc']) / 2.0
    else:
        profile_settings['t_end'] = (traj_length / V_mx) + (V_mx / acc_equiv)
        x = (2.0 / COEF_PROFILE_PARABOLIC) * V_mx
        profile_settings['t_acc'] = x / G_mx
        profile_settings['t_dec'] = profile_settings['t_end'] - (x / G_mxd)

    profile_settings['s_acc'] = G_mx * profile_settings['t_acc'] * profile_settings['t_acc'] / 2.0
    profile_settings['s_dec'] = profile_settings['s_acc'] + (profile_settings['v_max'] * (profile_settings['t_dec'] - profile_settings['t_acc']))
    profile_settings['s_end'] = traj_length

    profile_settings['t_end'] += t_suppl
    profile_settings['t_acc'] += t_suppl
    profile_settings['t_dec'] += t_suppl

    return profile_settings

## Profile Generator
# This function generates the position and the curve id for a given time in a parabolic profile.
def generate_parabolic_profile(profile_data, time):
    ABSCISSE_START = 0  # replace with actual value
    ABSCISSE_INTERMEDIATE = 1  # replace with actual value
    ABSCISSE_FINAL = 2  # replace with actual value

    a_acc = profile_data['acc_max']
    a_dec = profile_data['dec_max']
    v_max = profile_data['v_max']
    t_zero = profile_data['t_zero']
    t_acc = profile_data['t_acc']
    t_dec = profile_data['t_dec']
    t_end = profile_data['t_end']
    s_acc = profile_data['s_acc']
    s_dec = profile_data['s_dec']
    s_end = profile_data['s_end']

    if time <= t_zero:
        curv_id = ABSCISSE_START
        s_curv = 0
    elif time <= t_acc:
        curv_id = ABSCISSE_INTERMEDIATE
        s_curv = 0.5 * a_acc * (time - t_zero) ** 2
    elif time <= t_dec:
        curv_id = ABSCISSE_INTERMEDIATE
        s_curv = s_acc + v_max * (time - t_acc)
    elif time < t_end:
        curv_id = ABSCISSE_INTERMEDIATE
        t_acc1 = time - t_dec
        s_curv = s_dec + v_max * t_acc1 - 0.5 * a_dec * t_acc1 ** 2
    elif time >= t_end:
        t_acc1 = t_end - t_dec
        s_curv = s_end
        curv_id = ABSCISSE_FINAL

    return s_curv, curv_id

# Interpolation
# This function interpolates the position of a point in a parabolic profile.
def interpolate_position(x_start, y_start, z_start, s_curv, curv_id, abscisse_intermediate,vect_dir):
    Point_start = [x_start, y_start, z_start]
    op_dim = len(Point_start)
    data_tCoord = [0.0, 0.0, 0.0]#{'Desired_op': [0]*op_dim}

    if True: #curv_id == 1:
        s_curv_next = 0
        for i in range(op_dim):  # Python uses 0-based indexing
            #data_tCoord['Desired_op'][i] = Point_start[i] + (s_curv * vect_dir[i])
            data_tCoord[i] = Point_start[i] + (s_curv * vect_dir[i])
    else:
        for i in range(op_dim):  # Python uses 0-based indexing
            #data_tCoord['Desired_op'][i] = Point_start[i]
            data_tCoord[i] = Point_start[i]

    return data_tCoord #data_tCoord['Desired_op']

if __name__ == "__main__":
    # Test the functions
    traj_length, vect_dir = calculate_trajectory(0, 0, 0, 10, 10, 10)
    print("traj_length:", traj_length)
    print("vect_dir:", vect_dir)

    profile_parameters= solve_parabolic_profile(traj_length, 1, 1, 0, 0, 10, 0, 0)
    print("profile parameters:", profile_parameters)

    s_curv, curv_id = generate_parabolic_profile(profile_parameters, 1)
    print("s_curv:", s_curv)
    print("curv_id:", curv_id)

    interpolated_position = interpolate_position(0, 0, 0, s_curv, curv_id, 1, vect_dir)
    print("interpolated_position:",interpolated_position)