import math
from ControllerWithGUIMicroDelta.MicroDelta.setting_delta import *

class DeltaKinematics:
    def __init__(self, ArmLength, RodLength, BassTri, PlatformTri):
        self.PlatformTri = PlatformTri            # end effector
        self.BassTri = BassTri                     # base
        self.RodLength = RodLength
        self.ArmLength = ArmLength
        self.x = self.y = self.z = 0.0
        self.theta = [0.0, 0.0, 0.0]
        self.pi = math.pi
        self.tan30 = math.tan(math.radians(30))
        self.tan60 = math.tan(math.radians(60))
        self.sin30 = math.sin(math.radians(30))
        self.sin120 = math.sin(math.radians(120))
        self.cos120 = math.cos(math.radians(120))

    # forward kinematics: (thetaA, thetaB, thetaC) -> (x0, y0, z0)
    def forward(self, thetaA=None, thetaB=None, thetaC=None):
        if thetaA is None or thetaB is None or thetaC is None:
            return self._forward(self.theta[0], self.theta[1], self.theta[2])
        else:
            return self._forward(thetaA, thetaB, thetaC)

    def _forward(self, thetaA, thetaB, thetaC):
        self.x = self.y = self.z = 0.0

        t = (self.BassTri - self.PlatformTri) * self.tan30 / 2.0
        dtr = self.pi / 180.0

        thetaA *= dtr
        thetaB *= dtr
        thetaC *= dtr

        y1 = -(t + self.ArmLength * math.cos(thetaA))
        z1 = -self.ArmLength * math.sin(thetaA)

        y2 = (t + self.ArmLength * math.cos(thetaB)) * self.sin30
        x2 = y2 * self.tan60
        z2 = -self.ArmLength * math.sin(thetaB)

        y3 = (t + self.ArmLength * math.cos(thetaC)) * self.sin30
        x3 = -y3 * self.tan60
        z3 = -self.ArmLength * math.sin(thetaC)

        dnm = (y2 - y1) * x3 - (y3 - y1) * x2

        w1 = y1 * y1 + z1 * z1
        w2 = x2 * x2 + y2 * y2 + z2 * z2
        w3 = x3 * x3 + y3 * y3 + z3 * z3

        # x = (a1*z + b1)/dnm
        a1 = (z2 - z1) * (y3 - y1) - (z3 - z1) * (y2 - y1)
        b1 = -((w2 - w1) * (y3 - y1) - (w3 - w1) * (y2 - y1)) / 2.0

        # y = (a2*z + b2)/dnm;
        a2 = -(z2 - z1) * x3 + (z3 - z1) * x2
        b2 = ((w2 - w1) * x3 - (w3 - w1) * x2) / 2.0

        # a*z^2 + b*z + c = 0
        aV = a1 * a1 + a2 * a2 + dnm * dnm
        bV = 2.0 * (a1 * b1 + a2 * (b2 - y1 * dnm) - z1 * dnm * dnm)
        cV = (b2 - y1 * dnm) * (b2 - y1 * dnm) + b1 * b1 + dnm * dnm * (z1 * z1 - self.RodLength * self.RodLength)

        # discriminant
        dV = bV * bV - 4.0 * aV * cV
        if dV < 0.0:
            return "non_existing_povar_error", self.x, self.y, self.z  # non-existing position. return error, x, y, z

        self.z = -0.5 * (bV + math.sqrt(dV)) / aV
        self.x = (a1 * self.z + b1) / dnm
        self.y = (a2 * self.z + b2) / dnm

        return "no_error", self.x, self.y, self.z

    # inverse kinematics
    def inverse(self, x=None, y=None, z=None):
        if x is None or y is None or z is None:
            return self._inverse(self.x, self.y, self.z)
        else:
            return self._inverse(x, y, z)
        
    # helper functions, calculates angle thetaA (for YZ-pane)
    def delta_calcAngleYZ(self, x0, y0, z0):
        y1 = -0.5 * self.tan30 * self.BassTri  # f/2 * tan(30 deg)
        y0 -= 0.5 * self.tan30 * self.PlatformTri  # shift center to edge

        # z = a + b*y
        aV = (x0 * x0 + y0 * y0 + z0 * z0 + self.ArmLength * self.ArmLength - self.RodLength * self.RodLength - y1 * y1) / (2.0 * z0)
        bV = (y1 - y0) / z0

        # discriminant
        dV = -(aV+bV*y1)*(aV+bV*y1)+self.ArmLength*(bV*bV*self.ArmLength+self.ArmLength) 
        if (dV < 0):
            return "error_angle", -1000 #// non-existing povar.  return error, theta

        yj = (y1 - aV*bV - math.sqrt(dV))/(bV*bV + 1) # // choosing outer povar
        zj = aV + bV*yj
        angle = math.atan2(-zj,(y1 - yj)) * 180.0/math.pi

        return "no_error", angle;  #// return error, theta

    def _inverse(self, x0, y0, z0):
        error1, self.theta[0] = self.delta_calcAngleYZ(x0,y0,z0)
        error2, self.theta[1] = self.delta_calcAngleYZ(x0*self.cos120 + y0*self.sin120, y0*self.cos120-x0*self.sin120, z0)
        error3, self.theta[2] = self.delta_calcAngleYZ(x0*self.cos120 - y0*self.sin120, y0*self.cos120+x0*self.sin120, z0)
        if (error1 != "no_error") & (error2 != "no_error") & (error3 != "no_error"):
            return "error_angle", self.theta[0], self.theta[1], self.theta[2]
        
        return "no_error", self.theta[0], self.theta[1], self.theta[2]

if __name__ == "__main__":
    #testing
    # Initialize the DeltaKinematics object
    # Replace these values with your actual values
    a = 30
    b = 45
    c = 60
    delta = DeltaKinematics(L,l,Sb,Sp)

    # Test forward kinematics
    status, x, y, z = delta.forward( 30,45,60)
    print(f"Forward kinematics result: {status}, {x}, {y}, {z}")

    # Test inverse kinematics
    # Use the result of the forward kinematics as the input
    status2, thetaA, thetaB, thetaC = delta.inverse(x, y, z)
    print(f"Inverse kinematics result: {status2}, {thetaA}, {thetaB}, {thetaC}")

    # The output angles should be close to the original input angles
    # Convert them back to degrees for comparison
    print(f"Original angles: {a}, {b}, {c}")
    print(f"Recovered angles: {thetaA}, {thetaB}, {thetaC}")