import sys

from ControllerWithGUIMicroDelta.MicroDelta.motor import motor
from ControllerWithGUIMicroDelta.MicroDelta.geometric_model import DeltaKinematics
from ControllerWithGUIMicroDelta.MicroDelta.trajectory_generation import calculate_trajectory, solve_parabolic_profile, generate_parabolic_profile, interpolate_position
from ControllerWithGUIMicroDelta.settings import *

import asyncio
import can
from can.notifier import MessageRecipient
from typing import List
import struct
import math as m
import time as T

class delta_robot():
    def __init__(self, serial_channel=None) -> None:
        self.serial_channel = serial_channel if serial_channel is not None else SERIAL_CHANNEL
        self.serial_channel_loaded_and_valid = None

        self.position = [0.0, 0.0, 0.0]
        self.theta = [0.0, 0.0, 0.0]
        self.target = [0.0, 0.0, -250.0]
        self.current = [0.0, 0.0, 0.0]
        self.timeVelocityCurrent = 0.0
        self.timePosition = 0.0
        self.vmax = 10.0
        self.amax = 5.0
        self.dmax = 5.0
        self.motors = [motor(1), motor(2), motor(3)]
        self.state = "Waiting"
        self.kinematics = DeltaKinematics(L,l,Sb,Sp)

        self.xTarget = 0.0
        self.yTarget = 0.0
        self.zTarget = -250.0
        self.message = ""
        self.message2print = False
        self.newTarget = False
        self.newTargetAngle = False
        self.motorControlID = 0
        self.targetAngle = DEFAULT_ANGLE
        self.trajectoryChoice = 0
        self.newParams = False
        self.PID = {'kp': DEFAULT_KP,
                    'ki': DEFAULT_KI,
                    'kd': DEFAULT_KD,
                    'arw': DEFAULT_ARW}
        self.run = True

        self.flag = {"message2print": False,
                     "newTarget": False,
                     "newAngle": False,
                     "newParams": False}

    def recieve_message(self, msg: can.Message):

        motor_id = (msg.arbitration_id & 0x0F0)>>4
        message_id = msg.arbitration_id & 0x00F

        if message_id == 2:
            _,self.current[motor_id-1] = self.CAN_Message_to_U_I(msg, arbitration_id=msg.arbitration_id)
            self.timeVelocityCurrent = msg.timestamp
        elif message_id == 3:
            self.theta[motor_id-1] = self.CAN_Message_to_Position(msg, arbitration_id=msg.arbitration_id)
            self.timePosition = msg.timestamp

    async def can_comunication(self):
        try:
            bus = can.Bus(bustype='slcan', channel=self.serial_channel, bitrate=500_000)
        except Exception as e:
            print(f"Error opening serial channel: {e}")
            self.serial_channel_loaded_and_valid = False
        else:
            with bus:
                self.serial_channel_loaded_and_valid = True

                reader = can.AsyncBufferedReader()
                logger = can.Logger(LOG_FILE)

                listeners: List[MessageRecipient] = [
                    self.recieve_message,
                    reader,
                    logger,
                ]

                loop = asyncio.get_running_loop()
                notifier = can.Notifier(bus, listeners, loop=loop)

                while self.run:
                    await self.FSM(bus)
                    await asyncio.sleep(0.01)

    async def FSM(self, bus: can.Bus):
        if self.state == "Transparent":
            self.print_gui(f"Sending Transparent cmd")
            for motor in self.motors:
                msg2send = self.rest_to_CAN_Message(motor.canId)
                bus.send(msg2send)
            self.state = "Waiting"
        elif self.state == "Homing":
            for motor in self.motors:
                msg2send = self.homing_to_CAN_Message(motor.canId)
                self.print_gui(f"Homing motor {motor.Id}")
                bus.send(msg2send)
                await asyncio.sleep(1)
            self.state = "Waiting"
        elif self.state == "Position":
            if self.newTarget:
                self.newTarget = False
                await self.movement(bus, self.target[0], self.target[1], self.target[2])
        elif self.state == "Home":
            await self.movement(bus, DEFAULT_X_TARGET, DEFAULT_Y_TARGET, DEFAULT_Z_TARGET)
            self.state = "Waiting"
        elif self.state == "Angle":
            if self.newTargetAngle:
                if self.trajectoryChoice == 0:
                    msg2send = self.Position_to_CAN_Message(self.targetAngle, self.motors[self.motorControlID].canId)
                    bus.send(msg2send)
                elif self.trajectoryChoice == 1:
                    pass
        elif self.state == "Points":
            if self.list_check(LIST_POINT):
                for index, point in enumerate(LIST_POINT):
                    await self.movement(bus, point[0], point[1], point[2])
                    self.print_gui(f'Point : {index}')
                self.state = "Transparent"
            else:
                self.print_gui(f'List outside range')
                await asyncio.sleep(2)
                self.state = "Transparent"

        elif self.state == "Params":
            if self.newParams == True:
                motor = self.motors[2]
                msg2send = self.PID_to_CAN_Message(self.PID['kp'],self.PID['ki'],self.PID['kd'],self.PID['arw'],motor.canId)
                self.print_gui(f"Set PID {motor.Id}")
                bus.send(msg2send)
                await asyncio.sleep(0.01)
                # for motor in self.motors:
                #     msg2send = self.PID_to_CAN_Message(self.PID['kp'],self.PID['ki'],self.PID['kd'],self.PID['arw'],motor.canId)
                #     self.print_gui(f"Set PID {motor.Id}")
                #     bus.send(msg2send)
                #     await asyncio.sleep(0.01)


    async def test_postion(self, bus: can.Bus, motorId: int, communicationId: int):
        target = self.theta[motorId-1]
        print(target)
        msg2send = self.Position_to_CAN_Message(target, communicationId)
        print(msg2send)
        bus.send(msg2send)
        await asyncio.sleep(5)

    async def test_trajectoire(self, bus: can.Bus, motorId: int, comunicationId: int, target: float):
        pos = self.theta[motorId-1]
        tajectory_length = abs(target-self.theta[motorId-1])
        vect_dir = [m.copysign(1.0, (target-self.theta[motorId-1])), 0.0, 0.0]
        profile = solve_parabolic_profile(tajectory_length, self.amax, self.dmax, V_init=0.0, V_fin=0.0, V_mx=self.vmax, Tj=0.0, t_suppl=0)
        time = 0.0
        while time < profile['t_end']:
            s_curv, curv_id = generate_parabolic_profile(profile,time)
            #data = interpolate_position(self.position[0], self.position[1], self.position[2], s_curv, curv_id, 0.0, vector_director)
            data = interpolate_position(pos, 0.0, 0.0, s_curv, curv_id, 0.0, vect_dir)
            msg2send = self.Position_to_CAN_Message(data[0], comunicationId)
            #print(msg2send)
            bus.send(msg2send)
            #angle = getAnglesDegreesFromPosition(tuple(data), min_theta=-45, max_theta=80, silent=True)
            #print(angle)
            time += 0.01
            await asyncio.sleep(0.01)

    async def movement(self, bus: can.Bus, xtarget, ytarget, ztarget):
        self.print_gui("Begin movement")
        xCurrent = self.position[0]
        yCurrent = self.position[1]
        zCurrent = self.position[2]
        #xtarget = self.target[0]
        #ytarget = self.target[1]
        #ztarget = self.target[2]
        trajectory_length, vector_director = calculate_trajectory(xCurrent, yCurrent, zCurrent, xtarget, ytarget, ztarget)
        profile = solve_parabolic_profile(trajectory_length, self.amax, self.dmax, V_init=0.0, V_fin=0.0, V_mx=self.vmax, Tj=0.0, t_suppl=0)
        time = 0.0
        while time < profile['t_end']:
            #timeMesured = T.time()
            s_curv, curv_id = generate_parabolic_profile(profile,time)
            #data = interpolate_position(self.position[0], self.position[1], self.position[2], s_curv, curv_id, 0.0, vector_director)
            data = interpolate_position(xCurrent, yCurrent, zCurrent, s_curv, curv_id, 0.0, vector_director)
            #print(data)
            status, theta1, theta2, theta3 = self.kinematics.inverse(x=data[0], y=data[1], z=data[2])
            #print(f'Theta1 {theta1}, Theta2 {theta2}, Theta3 {theta3}')
            msg2send1 = self.Position_to_CAN_Message(theta1,0x011)
            msg2send2 = self.Position_to_CAN_Message(theta2,0x021)
            msg2send3 = self.Position_to_CAN_Message(theta3,0x031)
            bus.send(msg2send1)
            bus.send(msg2send2)
            bus.send(msg2send3)
            #angle = getAnglesDegreesFromPosition(tuple(data), min_theta=-45, max_theta=80, silent=True)
            #print(angle)
            #timeSend = T.time()
            #print(timeMesured-timeSend)
            time += 0.01
            await asyncio.sleep(0.01)
        self.print_gui("Motion done")

    def print_gui(self, text: str):
        self.message = text
        self.message2print = True

    def list_check(self, list_point):
        for point in list_point:
            if (X_MIN<=point[0]<=X_MAX) & (Y_MIN<=point[1]<=Y_MAX) & (Z_MIN<=point[2]<=Z_MAX):
                pass
            else:
                return False
        return True

    ##Proper functions
    def rest_to_CAN_Message(self, arbitration_id: int) :
        rest = [0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00]
        msg = can.Message(arbitration_id=arbitration_id, data=rest, is_extended_id=False)
        return msg

    def homing_to_CAN_Message(self, arbitration_id: int):
        homming = [0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00]

        # Create CAN Message
        msg = can.Message(arbitration_id=arbitration_id, data=homming, is_extended_id=False)

        return msg

    def Position_to_CAN_Message(self, position, arbitration_id: int):
        # Multiply position by factor
        position *= 10000

        # Convert to int32
        position = int(position)

        # Convert to hexadecimal and cast to bytes
        position_bytes = struct.pack('!I', position)

        # Create CAN message
        msg = can.Message(arbitration_id=arbitration_id,
                        data=[0x03] + list(position_bytes),
                        is_extended_id=False)

        return msg

    def CAN_Message_to_Position(self, msg: can.Message, arbitration_id: int):
        # Check if the arbitration ID matches the sender_id
        if msg.arbitration_id != arbitration_id:
            return None

        # Extract the position bytes
        position_bytes = msg.data[0:4]

        # Convert the data from bytes to signed integer
        position = struct.unpack('!i', bytes(position_bytes))[0]

        # Divide the position by the factor
        position /= 10000

        return position

    def CAN_Message_to_U_I(self, msg: can.Message, arbitration_id: int):
        # Check if the arbitration ID matches the sender_id
        if msg.arbitration_id != arbitration_id:
            return None

        # Check if the data length is correct
        if len(msg.data) != 8:
            return None

        # Extract the voltage bytes and convert to integer
        voltage_bytes = msg.data[0:4]
        voltage = struct.unpack('!i', bytes(voltage_bytes))[0]

        # Extract the current bytes and convert to integer
        current_bytes = msg.data[4:8]
        current = struct.unpack('!i', bytes(current_bytes))[0]

        # Divide the voltage and current by the factor
        voltage /= 1000
        current /= 1000

        return voltage, current

    def PID_to_CAN_Message(self, Kp, Ki, Kd, arw, arbitration_id):
        # Multiply by factor
        Kp *= 100
        Ki *= 100
        Kd *= 100
        arw /= 12

        # Convert to uint16 and cast to bytes
        Kp_bytes = struct.pack('!H', int(Kp))
        Ki_bytes = struct.pack('!H', int(Ki))
        Kd_bytes = struct.pack('!H', int(Kd))
        arw_bytes = struct.pack('!B', int(arw))

        # Create CAN message
        msg = can.Message(arbitration_id=arbitration_id,
                        data=[0x06] + list(Kp_bytes) + list(Ki_bytes) + list(Kd_bytes) + list(arw_bytes),
                        is_extended_id=False)
        
        return msg