% ======================================================================= %
% Matlab script
% M.A. Habisreutinger 2015
% ======================================================================= %


function DiffusionEquilibriumFactorized2D()
  % M.A. Habisreutinger 2015
  
  %% 1.   PARAMETERS

  %  1.1  Mesh
  ndim = 2 ; L = [2 1] ; np = 2^7*L ; 

  %  1.2  Diffusion coefficient
  nu = 1e+00 ;

  %  1.3  Linear solver
  iterations = 1e+02 ; tolerance = 1e-06 ;


  %% 2.   COMPUTATION
  
  %  2.0  Timing
  tstart = clock ;

  %  2.1  Nodal distributions
  for idim = 1 : ndim
    xi{idim} = linspace(-L(idim)/2,+L(idim)/2,np(idim)) ;
    h(idim)  = L(idim)/(np(idim)-1) ;
  end

  %  2.2  Mesh
  [X{2},X{1}] = meshgrid(xi{2},xi{1}) ;

  %  2.3  Discrete operators
  for idim = 1 : ndim
    e = ones(np(idim),1) ;
    % Central finite differences (2nd order)
    Mi{idim} = spdiags(e,0,np(idim),np(idim)) ;
    Ai{idim} = spdiags(-(nu/h(idim)^2)*[+1*e -2*e +1*e],-1:1,np(idim),np(idim)) ;
    % Dirichlet boundary condtitions
    Ai{idim}(1,:) = 0 ; Ai{idim}(np(idim),:)        = 0 ;
    Ai{idim}(1,1) = 1 ; Ai{idim}(np(idim),np(idim)) = 1 ;
  end

  %  2.4  Tensorisation for a 2D problem
  M = @(v)Mfun(Mi,v,np) ;
  A = @(v)Afun(Ai,Mi,v,np) ;

  %  2.5  Exact solution
  k  = 2*pi./L ;
  U  = sin(k(1)*X{1}).*sin(k(2)*X{2}) ;
  
  %  2.6  Forcing term
  F  = (k(1)^2+k(2)^2)*nu*U ; 
  MF = feval(M,F) ;
  Mf = reshape(MF,prod(np),1) ;

  %  2.7  Solution of the discrete equations
  [uh,~,~,~,resvec] = gmres(A,Mf,[],tolerance,min(prod(np),iterations)) ;
  Uh = reshape(uh,np) ;

  
  %% 3.   DATA ANALYSIS

  %  3.1  Error norm
  eh = norm(feval(M,Uh-U))/norm(feval(M,U)) ;

  %  3.2  Display
  disp(['Error norm   : ',num2str(eh)])
  disp(['Elapsed time : ',num2str(etime(clock,tstart))])
  
  %  3.3  Convergence plot
  figure(1) ; semilogy(resvec,'bo-') ; grid on ;
  xlabel('Iterations') ; ylabel('Residuals') ; title('Convergence history') ;
  
  %  3.4  Numerical solution plot
  figure(2) ; subplot(2,1,1) ;
  pcolor(X{1},X{2},Uh) ; shading flat ; colorbar ;
  axis equal ; axis tight ; caxis([min(min(U)) max(max(U))]) ; colormap('hot') ;
  xlabel('x') ; ylabel('y') ; title('Discrete solution') ;
  
  %  3.5  Analytical solution plot
  figure(2) ; subplot(2,1,2) ;
  pcolor(X{1},X{2},U) ; shading interp ; colorbar ;
  axis equal ; axis tight ; caxis([min(min(U)) max(max(U))]) ; colormap('hot') ;
  xlabel('x') ; ylabel('y') ; title('Exact solution') ;
  
end

% ======================================================================= %

function w = Mfun(Mi,v,np)
  % M.A. Habisreutinger 2015
  % w = [(My o Mx)]v
   
  V = reshape(v,np) ;
  W = Mi{1}*V*transpose(Mi{2}) ;
  w = reshape(W,prod(np),1) ;
  
end

% ======================================================================= %

function w = Afun(Ai,Mi,v,np)
  % M.A. Habisreutinger 2015
  % w = [(Ay o Mx)+(My o Ax)]v
   
  V = reshape(v,np) ;
  W = Ai{1}*V*transpose(Mi{2})+Mi{1}*V*transpose(Ai{2}) ;
  w = reshape(W,prod(np),1) ;
  
end

% ======================================================================= %