% ======================================================================= %
% Matlab script
% M.A. Habisreutinger 2015
% ======================================================================= %


function DiffusionEquilibriumAssembled3D()
  % M.A. Habisreutinger 2015
  
  %% 1.   PARAMETERS

  %  1.1  Mesh
  ndim = 3 ; L = [2 1 4] ; np = 2^5*L ; 

  %  1.2  Diffusion coefficient
  nu = 1e+00 ;

  %  1.3  Linear solver
  iterations = 1e+02 ; tolerance = 1e-06 ;

  %  1.4  Vtk output for Paraview (download at www.paraview.org)
  write = true ; filename = 'DiffusionEquilibrium3D' ; varname = 'uh' ;


  %% 2.   COMPUTATION

  %  2.0  Timing
  tstart = clock ;
  
  %  2.1  Nodal distributions
  for idim = 1 : ndim
    xi{idim} = linspace(-L(idim)/2,+L(idim)/2,np(idim)) ;
    h(idim)  = L(idim)/(np(idim)-1) ;
  end

  %  2.2  Mesh
  [X{2},X{1},X{3}] = meshgrid(xi{2},xi{1},xi{3}) ;
  for idim = 1 : ndim
    x(:,idim) = reshape(X{idim},prod(np),1) ;
  end

  %  2.3  Discrete operators
  for idim = 1 : ndim
    e = ones(np(idim),1) ;
    % Central finite differences (2nd order)
    Mi{idim} = spdiags(e,0,np(idim),np(idim)) ;
    Ai{idim} = spdiags(-(nu/h(idim)^2)*[+1*e -2*e +1*e],-1:1,np(idim),np(idim)) ;
    % Dirichlet boundary condtitions
    Ai{idim}(1,:) = 0 ; Ai{idim}(np(idim),:)        = 0 ;
    Ai{idim}(1,1) = 1 ; Ai{idim}(np(idim),np(idim)) = 1 ;
  end

  %  2.4  Tensorisation for a 3D problem
  M = kron(Mi{3},kron(Mi{2},Mi{1})) ;
  A = kron(Ai{3},kron(Mi{2},Mi{1})) ...
    + kron(Mi{3},kron(Ai{2},Mi{1})) ...
    + kron(Mi{3},kron(Mi{2},Ai{1})) ;

  %  2.5  Exact solution
  k = 2*pi./L ;
  u = sin(k(1)*x(:,1)).*sin(k(2)*x(:,2)).*sin(k(3)*x(:,3)) ;
  
  %  2.6  Forcing term
  f  = (k(1)^2+k(2)^2+k(3)^2)*nu*u ;
  Mf = M*f ; 
  
  %  2.7  Solution of the discrete equations
  [uh,~,~,~,resvec] = gmres(A,Mf,[],tolerance,min(prod(np),iterations)) ;

  
  %% 3.   DATA ANALYSIS

  %  3.1  Error norm 
  eh = norm(M*(uh-u))/norm(M*u) ;

  %  3.2  Display
  disp(['Error norm   : ',num2str(eh)])
  disp(['Elapsed time : ',num2str(etime(clock,tstart))])
  
  %  3.3  Convergence plot
  figure(1) ; semilogy(resvec,'bo-') ; grid on ;
  xlabel('Iterations') ; ylabel('Residuals') ; title('Convergence history') ;
  
  %  3.4  Vtk output
  if write
    WriteVtkOutput(filename,varname,xi,uh,np) ;
  end

end

% ======================================================================= %

function WriteVtkOutput(filename,varname,xi,v,np)
  % M.A. Habisreutinger 2015
  
  % File
  file = fopen(strcat(filename,'.vtk'),'w') ;

  % Header
  fprintf(file,'%s\n','# vtk DataFile Version 3.1') ;
  fprintf(file,'%s\n','Export Q') ;
  fprintf(file,'%s\n','ASCII') ;
  fprintf(file,'%s\n','DATASET RECTILINEAR_GRID') ;
  
  fprintf(file,'%s ','DIMENSIONS') ;
  fprintf(file,'%u %u %u\n',np) ;
  
  fprintf(file,'%s ','X_COORDINATES') ;
  fprintf(file,'%u ',np(1)) ;
  fprintf(file,'%s\n','FLOAT') ;
  
  fprintf(file,'%15.5f',xi{1}) ; fprintf(file,'\n') ;
  
  fprintf(file,'%s ','Y_COORDINATES') ;
  fprintf(file,'%u ',np(2)) ;
  fprintf(file,'%s\n','FLOAT') ;

  fprintf(file,'%15.5f',xi{2}) ; fprintf(file,'\n') ;
    
  fprintf(file,'%s ','Z_COORDINATES') ;
  fprintf(file,'%u ',np(3)) ;
  fprintf(file,'%s\n','FLOAT') ;

  fprintf(file,'%15.5f',xi{3}) ; fprintf(file,'\n') ;
  
  fprintf(file,'%s ','POINT_DATA') ;
  fprintf(file,'%u\n',prod(np)) ;
  
  % Fields
  v = reshape(v,prod(np),1) ;
  fprintf(file,'%s ','SCALARS') ;
  fprintf(file,'%s ',varname) ;
  fprintf(file,'%s\n','FLOAT') ;
  fprintf(file,'%s\n','LOOKUP_TABLE default') ;
  fprintf(file,'%15.5f',v') ; fprintf(file,'\n') ;
  
  fclose(file) ;
  
end

% ======================================================================= %