% ======================================================================= %
% Matlab script
% M.A. Habisreutinger 2015
% ======================================================================= %


%% 0.   CLEAN & FORMAT
clear all ; close all ;
format compact ; format short e ;


%% 1.   PARAMETERS

%  1.1  Mesh
ndim = 2 ; L = [2 1] ; np = 2^7*L ; 

%  1.2  Diffusion coefficient
nu = 1e+00 ;

%  1.3  Linear solver
iterations = 1e+02 ; tolerance = 1e-06 ;


%% 2.   COMPUTATION

%  2.0  Timing
tstart = clock ;

%  2.1  Nodal distributions
for idim = 1 : ndim
  xi{idim} = linspace(-L(idim)/2,+L(idim)/2,np(idim)) ;
  h(idim)  = L(idim)/(np(idim)-1) ;
end

%  2.2  Mesh
[X{2},X{1}] = meshgrid(xi{2},xi{1}) ;
for idim = 1 : ndim
  x(:,idim) = reshape(X{idim},prod(np),1) ;
end

%  2.3  Discrete operators
for idim = 1 : ndim
  e = ones(np(idim),1) ;
  % Central finite differences (2nd order)
  Mi{idim} = spdiags(e,0,np(idim),np(idim)) ;
  Ai{idim} = spdiags(-(nu/h(idim)^2)*[+1*e -2*e +1*e],-1:1,np(idim),np(idim)) ;
  % Dirichlet boundary condtitions
  Ai{idim}(1,:) = 0 ; Ai{idim}(np(idim),:)        = 0 ;
  Ai{idim}(1,1) = 1 ; Ai{idim}(np(idim),np(idim)) = 1 ;
end

%  2.4  Tensorisation for a 2D problem
M = kron(Mi{2},Mi{1}) ;
A = kron(Mi{2},Ai{1})+kron(Ai{2},Mi{1}) ;

%  2.5  Exact solution
k = 2*pi./L ;
u = sin(k(1)*x(:,1)).*sin(k(2)*x(:,2)) ;

%  2.6  Forcing term
f  = (k(1)^2+k(2)^2)*nu*u ;
Mf = M*f ;

%  2.7  Solution of the discrete equations
[uh,~,~,~,resvec] = gmres(A,Mf,[],tolerance,min(prod(np),iterations)) ;


%% 3.   DATA ANALYSIS

%  3.1  Error norm 
eh = norm(M*(uh-u))/norm(M*u) ;

%  3.2  Display
disp(['Error norm   : ',num2str(eh)])
disp(['Elapsed time : ',num2str(etime(clock,tstart))])

%  3.3  Convergence plot
figure(1) ; semilogy(resvec,'bo-') ; grid on ;
xlabel('Iterations') ; ylabel('Residuals') ; title('Convergence history') ;

%  3.4  Numerical solution plot
figure(2) ; subplot(2,1,1) ;
pcolor(reshape(x(:,1),np),reshape(x(:,2),np),reshape(uh,np)) ; 
shading flat ; colorbar ;
axis equal ; axis tight ; caxis([min(u) max(u)]) ; colormap('hot') ;
xlabel('x') ; ylabel('y') ; title('Discrete solution') ;

%  3.5  Analytical solution plot
figure(2) ; subplot(2,1,2) ;
pcolor(reshape(x(:,1),np),reshape(x(:,2),np),reshape(u,np)) ;
shading interp ; colorbar ;
axis equal ; axis tight ; caxis([min(u) max(u)]) ; colormap('hot') ;
xlabel('x') ; ylabel('y') ; title('Exact solution') ;

% ======================================================================= %