"""
This file contains the specific functions to solve linear wavev equation
in 1D using a flux-limited scheme.
"""

import numpy as np
from Linwave import *
from helpers import extend,FluxLimit

## 1D ##
###################################################################################
def LinwaveFLrhs1D(x, u, h, k, maxvel):

    """Purpose: Evaluate right hand side for linear wave equation using a
                flux limited scheme"""
    N = len(x)

    # Chose flux limiter - 0:LF; 1:CO; 2:Koren; 3:Sweby; 4:OSPRE; 5:van Leer
    typ = 5
    beta = 1.5

    # Boundary conditions 
    xe, ue = extend(x, u, 1, "P", 0, "P", 0) # Periodic boundary conditions
    #xe, ue = extend(x, u, 1, "D", 2, "N", 0) # Constant boundary conditions

    # Compute indicator function and define flux limiter
    r = (ue[1:N+1] - ue[:N])/(ue[2:N+2] - ue[1:N+1])
    xe,re = extend(x,r,1,"N",0,"N",0)
    rm = 1.0/re

    phiL = FluxLimit(re[:N], typ, beta)
    phiR = FluxLimit(re[1:N+1], typ, beta)

    # Compute left flux in cell - Change numerical flux here
    Fluxlow = LinwaveLF(ue[:N],ue[1:N+1],maxvel)
    Fluxhigh = LinwaveLW( ue[:N],ue[1:N+1],k/h)
    FluxL = Fluxlow + phiL*(Fluxhigh-Fluxlow)

    # Compute right flux in cell - Change numerical flux here
    Fluxlow = LinwaveLF(ue[1:N+1],ue[2:N+2],maxvel)
    Fluxhigh = LinwaveLW( ue[1:N+1],ue[2:N+2],k/h)
    FluxR = Fluxlow + phiR*(Fluxhigh-Fluxlow)

    # Compute RHS
    du = - (FluxR-FluxL)/h
    return du

def LinwaveFL1D(x, u, h, CFL, FinalTime):
    """Purpose  : Integrate 1D Linwave equation until FinalTime using a flux limited scheme.
    """   
    t = 0.0
    timestep = 0

    # Set timestep
    k = CFL*h
    maxvel = 1.0

    while t < FinalTime:
        k = min(FinalTime-t, k)
        # Update solution
        u += k*LinwaveFLrhs1D(x,u,h,k,maxvel)
        t +=k
        timestep += 1
        
    return u