"""
Contents:
- Nonvisc1
- Nonvisc2
- HeatFlux
- HeatDGrhs1D
- FilterFD
- FourierD
- FourierF
- GegenbauerP
- GegenbauerGQ
- GegenbauerPade
- GegenbauerRecon
- FourierPade 
- SingularFourierPade
- Entvisc
"""

import numpy as np
from helpers import extendDG
from scipy.integrate import quad
from scipy.special import betainc,gamma,jv
from scipy.linalg import toeplitz

def Nonvisc1(x,u,iV,m,N,h,nu0,kappa):
    """Purpose: Compute nonlinear viscosity following Persson and Peraire (2006)"""

    nu = np.zeros((m+1,N))
    S = np.zeros(N)
    nuh = np.zeros(N)
    onev = np.ones(m+1)

    # Extract coefficients and compute smoothness measure
    uh = np.dot(iV,u)
    S = uh[m,:]**2/np.sum(uh*uh)
    se = np.log(S) 

    # Compute elementwise viscosity
    s0 = np.log(1.0/m**4)
    nu1 = ((s0-kappa)<=se)*(se<=(s0+kappa)) 
    nu2 = (se>(s0+kappa))
    nuh = nu0*h/m*(nu1/2*(1.0+np.sin(np.pi*(se-s0)/(2.0*kappa))) + nu2)

    # Compute continuous viscosity
    nue = np.zeros(N+2)
    nue = np.concatenate(([nuh[0]], nuh, [nuh[N-1]]))

    maxL = np.fmax(nue[:N],nue[1:N+1])
    maxR = np.fmax(nue[1:N+1],nue[2:N+2])

    nu = np.outer(onev,maxL) + (x-np.outer(onev,x[0,:]))/h*np.outer(onev,(maxR-maxL))

    return nu



def Nonvisc2(x,u,iV,m,N,h,nu0):
    """Purpose: Compute nonlinear viscosity following Kloeckner et al (2013)""" 
    nu = np.zeros((m+1,N))
    S = np.zeros(N)
    nuh = np.zeros(N)
    onev = np.ones(m+1)
    eps0 = 1e-10

    # Special case of m=1,2
    if (m<3):
        nuh = h/m*np.outer(nu0, np.ones(N))
    else:
        # Extract coefficients and compute smoothness measure
        uh = np.dot(iV,u)
        uh1 = uh[1:m+1,:]

        # Adjust for scaling 
        bh = np.arange(1,m+1).astype(float)**(-m)/np.sqrt(np.sum(np.arange(1,m+1).astype(float)**(-2*m)))
        ut = np.sqrt(uh1*uh1 + (bh*bh)*np.sum(uh1*uh1))
       
        # Adjust for non-monotone decay
        ub = ut
        for i in range(m):
            ub[i,:] = np.max(np.abs(ut[i:m,:]),axis=0)
       
        # Compute decay estimate by least squares fit
        b1 = np.log(np.arange(1,m+1))
        h1 = -np.sum(b1)
        h2 = -b1*np.log(ub+eps0)
        A = np.array(([m, h1],[h1,h1**2]))
        b = np.array((-h1*np.ones(N),h2))
        coef = np.linalg.solve(A,b)
     
        # Compute elementwise viscosity
        nu1 = (coef[1,:]<=1.0)
        nu2 = (coef[1,:]>1.0)*(coef[1,:]<3.0)

        nuh = nu0*h/m*(nu1 + nu2*(1.0-(coef[1,:]-1.0)/2.0))

    # Compute continuous viscosity
    nue = np.zeros((1,N+2))
    nue = np.concatenate(([nuh[0]], nuh, [nuh[N-1]]))
    maxL = np.fmax(nue[:N],nue[1:N+1])
    maxR = np.fmax(nue[1:N+1],nue[2:N+2])

    nu = np.outer(onev,maxL) + (x-np.outer(onev,x[0,:]))/h*(np.outer(onev,(maxR-maxL)))
    return nu


def HeatFlux(u,v,typ):
    """Purpose: Compute flux for heat equation.
             Type: 'L' =Upwind, 'C'=Central, 'R'=Downwind"""

    if typ == "L":
        flux = u
    elif typ == "C":
        flux = 0.5*(u+v)
    elif typ == "R":
        flux = v
    return flux


def HeatDGrhs1D(x,u,h,k,m,N,Ma,S,VtoE,nu):
    """Purpose  : Evaluate the RHS flux in 1D heat equation
                 using a DG method"""
    Imat = np.eye(m+1)
    ue = np.zeros((2,N+2))
    qe = np.zeros((2,N+2))

    # Extend data and assign boundary conditions
    ue = extendDG(u[VtoE[0],VtoE[1]], "D", 0, "D", 0) 

    # Compute numerical fluxes at interfaces
    fluxr = HeatFlux( ue[1,1:N+1], ue[0,2:N+2], "C")
    fluxl = HeatFlux( ue[1,:N], ue[0,1:N+1], "C")

    # Compute aux variable q
    qh = np.dot(np.transpose(S), u) - (np.outer(Imat[:,m], fluxr) - \
                np.outer(Imat[:,0], fluxl))
    q = nu*( np.linalg.solve(h/2*Ma, qh ) )

    # Extend data and assign boundary conditions
    qe = extendDG(q[VtoE[0],VtoE[1]], "N", 0, "N", 0)

    # Compute numerical fluxes at interfaces
    fluxr = HeatFlux( qe[1,1:N+1], qe[0,2:N+2], "C")
    fluxl = HeatFlux( qe[1,:N], qe[0,1:N+1], "C")

    rh = np.dot(np.transpose(S), u) - (np.outer(Imat[:,m], fluxr) - \
                np.outer(Imat[:,0], fluxl))

    rhsu = nu*( np.linalg.solve(h/2*Ma, rh ) )

    return rhsu


def FilterFD(M,p):
    """Purpose: Compute filter coefficients 0:M for filter of order p 
                and width 2M+1."""

    fc = np.zeros(M+1)

    # Exponential filter (p even)
    alpha = 10.0
    f = lambda x,n: np.exp(-alpha*(x/np.pi)**p)*np.cos(n*x)

    # Optimal filter of order p
    # f = lambda x,n: (1.0-betainc(p,p,x/np.pi))*np.cos(n*x)

    for m in range(M+1):
        fc[m] = quad(lambda x: f(x,m), 0.0, np.pi)[0]/np.pi

    return fc


def FourierD(N,q):
    """Purpose: Initialize q'th order Fourier differentiation matrix"""

    column = np.concatenate(([0], (-1)**(np.arange(1,2*N+1))/ \
                (2*np.sin(np.pi/(2*N+1)*(np.arange(1,2*N+1))))))

    v = np.concatenate(([0], np.arange(2*N,0,-1)))
    D = toeplitz( column, column[v] )**q

    return D

def FourierF(u,p,alpha,N0):
    """Purpose: Apply modal exponential filter in a Fourier spectral approach"""

    N = len(u)
    N2 = (0.5*(N-1))

    # Define filter function
    sigma = np.ones(N2+1)
    sigma[N0:int(N2)+1] = np.exp(-alpha*((np.arange(N0,N2+1)-N0)/(N2-N0))**(2*p))
    nvec = np.concatenate((sigma, np.flipud(sigma[1:])))
    Fu = np.real(np.fft.ifft(nvec*np.fft.fft(u)))

    return Fu


def GegenbauerP(r,lam,N):
    """Purpose: Evaluate Gegenbauer polynomial of type lambda > -1/2
         points r for order N and returns P[1:length(r))]
        Note   : They are normalized to be orthonormal."""

    CL = np.zeros((N+1, len(r)))

    # Initial values C_0(x) and C_1(x)
    gamma0 = np.sqrt(np.pi)*gamma(lam+0.5)/gamma(lam+1.0)
    CL[0,:] = 1.0/np.sqrt(gamma0)

    if ( N == 0):
        return CL[0,:]

    gamma1 = (lam+0.5)**2/(2.0*lam+2.0)*gamma0
    CL[1,:] = (2.0*lam+1.0)*r/(2.0*np.sqrt(gamma1))

    if (N == 1):
        return CL[N,:]

    # Repeat value in recurrence
    aold = 2.0/(2*lam+1.0)*np.sqrt((lam+0.5)**2/(2.0*lam+2.0))

    # Forward recurrence using the symmetry of the recurrence
    for i in range(1,N):
        h1 = 2.0*i+2*lam-1.0
        anew = 2.0/(h1+2)*np.sqrt( (i+1)*(i+2*lam)*(i+lam+0.5)**2/(h1+1.0)/(h1+3.0) )

        CL[i+1,:] = 1.0/anew*(-aold*CL[i-1,:] + r*CL[i,:])
        aold = anew

    return CL[N,:]


def GegenbauerGQ(lam, N):
    """Purpose: Compute the N'th order Gauss quadrature points, x, 
         and weights, w, associated with the Gegenbauer 
         polynomial, of type lambda > -1/2"""

    if N == 0:
        x = np.array([0.0])
        w = np.array([2.0])

        return x,w

    J = np.zeros((N+1,N+1))
    h1 = 2.0*np.arange(N+1) + 2.0*lam-1.0

    J = np.diag( 2.0/(h1[:N] + 2.0)*np.sqrt( np.arange(1,N+1)*\
                ( np.arange(1,N+1) + 2.0*lam-1.0 ) * ( np.arange(1,N+1) + lam-0.5)**2 / \
                  (h1[:N]+1.0) /(h1[:N]+3.0) ),1 )

    if 2.0*lam-1.0<10*np.finfo(float).eps:
        J[0,0] = 0.0
    if (np.abs(lam)<10*np.finfo(float).eps):
        J[0,1] = lam+0.5

    J = J + J.T

    # Compute quadrature by eigenvalue solve
    x,V = np.linalg.eig(J)

    # sort eigenvalues
    idx = x.argsort()  
    x = x[idx]
    V = V[:,idx]

    w = (V[0,:])**2*2.0**(2.0*lam)*gamma(lam+0.5)**2/gamma(2.0*lam+1.0)

    return x,w


def GegenbauerPade(b,r,lam,N,M,L):
    """Purpose: Express Gegenbauer expansion as Pade form to suppress
        Runge phenomenon in diagonal limit"""

    NQ = 2*N
    xG,wG = GegenbauerGQ(lam,NQ)

    # Evaluate Gegenbauer polynomials at nodes to order 2*N
    Cmat = np.zeros((NQ+1,N+1))

    for m in range(N+1):
        Cmat[:,m] = GegenbauerP(xG,lam,m)

    # Evaluate function at quadrature points
    u = np.dot(Cmat, b)

    # Set up coefficient matrix
    H = np.dot( Cmat.T, np.dot(np.diag(u*wG),Cmat))

    # Compute coefficient for q
    Hq = H[M+1:N+1,:L+1]
    Z = nullspace(Hq)
    q = Z[:,-1]
    q /= np.min(q)

    # Compute coefficient for p
    p = np.dot(H[:M+1,:L+1],q)

    # Evaluate Pade form at r
    Cmat = np.zeros((len(r),N+1))
    for m in range(N+1):
        Cmat[:,m] = GegenbauerP(r,lam,m)

    pG = np.dot( Cmat[:,:M+1], p)
    qG = np.dot(Cmat[:,:L+1], q)


    return pG/qG


def GegenbauerRecon(x,u,xp,xeval,N,Nc):
    """Purpose: Gegenbauer reconstruction of u with points of discontinuity in 
    vector xp. Recobstructed at xeval xp(1)/xp(end) = left/right point of domain
    Only Nc first terms are used."""

    Nseg = len(xp)-1
    Llen = xp[-1]-xp[0]

    xleval = len(xeval)
    uRecon = np.zeros(xleval)

    uhat = np.fft.fft(u)/(2.0*N+1)

    if (Nc<N):
        uhat[Nc+1:2*N+1-Nc] = 0.0

    # Set parameters for the Gegenbauer construction
    lam = np.floor(np.sqrt(Nc))
    M = np.floor(np.sqrt(Nc)).astype(int)

    # Postprocess each segment
    for ns in range(1,Nseg+1):
        a = xp[ns-1]
        b = xp[ns]
        epsh = (b-a)/(2.0*np.pi)
        deltah = (a+b)/(2.0*np.pi)

        idd = np.where( np.logical_and(xeval >= a, xeval <= b) )[0]

        xl = xeval[idd]
        xlen = len(idd)
        r = -1.0+2.0*(xl-a)/(b-a)

        Cmat = np.zeros((xlen, M+1))
        b = np.zeros(M+1)

        for m in range(M+1):
            Cmat[:,m] = GegenbauerP(r,lam, m)
            gammaG = np.pi*2.0**(1.0-2.0*lam)*gamma(m+2*lam)/\
                            (gamma(m+1)*(m+lam)*gamma(lam)**2)

            for n in range(N+1):
                b[m] += uhat[n]*(2.0/(n*epsh*np.pi))**lam*\
                        jv(m+lam,n*epsh*np.pi)*np.exp(1.0j*np.pi*n*deltah)

                b[m] += uhat[2*N+1-n]*(2.0/(-n*epsh*np.pi))**lam*\
                        jv(m+lam,-n*epsh*np.pi)*np.exp(-1.0j*np.pi*n*deltah)


            b[m] = np.sqrt(gammaG)*gamma(lam)*(1.0j)**m*(m+lam)*b[m]

        gammaG0 = np.pi*2**(1.0-2.0*lam)*gamma(2.0*lam)/(lam*gamma(lam)**2)
        b[0] += uhat[0]*np.sqrt(gammaG0)

        uRecon[idd] = np.dot(Cmat,b)

    return uRecon


def FourierPade(x,u,N0,Nc,M,L):
    """Purpose: Apply Fourier-Pade postprocessing to periodic function with
   p_M(x)/q_L(x) = u_Nc(x) + x^(Nc+1). u_N is assumed real
 Approach follows Driscoll and Fornberg (2001).
 NOTE: M+L=Nc is assumed"""

    # Extract Nc Fourier coefficients and compute denominator
    uh = np.fft.fft(u)/(2*N0+1)
    uhat = uh[:Nc+1]
    N = Nc

    Cq = uhat[M+1:N+1]
    Rq = np.concatenate((uhat[np.arange(M+1,max(-1,M-L),-1)],\
                         np.zeros(L-M-1)))

    Z = nullspace( toeplitz(Cq,Rq))
    qp = Z[:,-1]
    qp = qp/qp[np.min(np.where(qp)[0])]
    qm = np.conj(qp)

    # Compute numerator
    Cq = uhat[:M+1]
    Cq[0] /= 2.0
    Rq = 1.0j*np.zeros(M+1)
    Rq[0] = Cq[0]
    A = toeplitz(Cq,Rq)

    pp = np.dot(A,qp[:M+1])
    pm = np.conj(pp)

    # Evaluate Pade-Fourier approximation
    Nx = len(x)
    xl = 2.0*np.pi/Nx*np.arange(Nx)
    xp = np.exp(1.0j*xl)
    xm = np.exp(-1.0j*xl)

    Pu = np.polyval(np.flipud(pp),xp)/np.polyval(np.flipud(qp),xp) + \
            np.polyval( np.flipud(pm),xm)/np.polyval(np.flipud(qm),xm)


    return Pu


def SingularFourierPade(x,u,z,N0,Nc):
    """Purpose: Apply Singular Fourier-Pade postprocessing to periodic function 
    p_M(x)/q_L(x) + r_R(x)/Q_L(x) = u_Nc(x) + x^(Nc+1). u_N is assumed real
    Approach follows Driscoll and Fornberg (2001) and
    padelog.m (Driscoll, MathWorks File Exchange, 2006)"""

    # Extract Nc Fourier coefficients and compute denominator
    uh = np.fft.fft(u)/(2*N0+1)
    uhat = uh[:Nc+1]
    N = Nc

    # Determine parameters and location of discontinuities
    Nx = len(x)
    xl = 2.0*np.pi/Nx*np.arange(Nx)
    xp = np.exp(1.0j*xl)
    xm = np.exp(-1.0j*xl)

    xmin = np.min(x)
    xmax = np.max(x)
    xi = (z-xmin)/(xmax-xmin)*2.0*np.pi
    xip = np.exp(1.0j*xi)
    xim = np.exo(-1.0j*xi)
    mm = len(z)

    # Setting orders of polynomials -- these can be adjusted
    L = np.ceil((N-mm)/(mm+1.5))-1
    s = np.floor((N-mm-L)/(mm+1.0))+1
    R = s*np.ones(mm)
    M = N-mm-L-np.sum(R)

    # Taylor coeffs of log terms
    k = np.arange(1,N+1)
    ls = []
    for s in range(1,mm+1):
        ls.append(np.concatenate(( [0], -1.0/(k*xsp[s-1]**k) )) ) 

    # The polynomials Q and R are found from the highest order coeff
    row = np.concatenate((uhat[np.arange(M+1,max(-1,M-L),-1)],\
                         np.zeros(L-M-1)))

    Cq = toeplitz(uhat[M+1:N+1],row)
    Lp = []
    for s in range(1,mm+1):
        row = np.concatenate((lp[s-1])[np.arange(M+1,max(-1,M-R[s-1]),-1)],\
                np.zeros(R[s-1]-M-1))
        Lp[s-1] = toeplitz( (lp[s-1])[M+1:N+1],row )

    # Find a vector v satisfying [Cq -Lp{1} ... -Lp{m}]*v = 0
    Z = nullspace( np.column_stack(( -Cq, Lp )) )
    qr = Z[:,-1]
    qr = qr/qr[np.min(np.where(qr)[0])]

    # Pull out polynomials
    qp = qr[:L+1]
    idx = L+1
    rp = []
    rs = []

    for s in range(1,mm+1):
        rp[s-1] = qr[idx+np.arange(R[s-1])]
        rm[s-1] = np.conj(rp[s-1])
        idx += R[s]+1

    # Remaining polynomial is found using low-order terms
    Cq = uhat[:M+1]
    Cq[0] = Cq[0]/2.0
    Rq = 1.0j*np.zeros(M+1)
    Rq[0] = Cq[0]

    A = toeplitz(Cq,Rq)

    pp = np.dot(A,qp[:M+1])

    for s in range(1,mm+1):
        Lp = toeplitz( (lp[s-1])[:M+1], \
                    np.concatenate(([lp[s-1]], np.zeros(R[s-1]) )) )
        pp -= np.dot(Lp, rp[s-1])

    pm = np.conj(pp)
    qm = np.conj(qp)

    # Evaluate singular Pade-Fourier form
    qpp = np.polyval(np.flipud(qp),xp)
    qmp = np.polyval(np.flipud(qm),xm)

    Pu = np.polyval(np.flipud(pp),xp)/qpp + \
         np.polyval(np.flipud(pm),xm)/qmp

    for s in range(1,mm+1):
        rpp = np.polyval(np.flipud(rp[s-1]),xp)
        rpm = np.polyval(np.flipud(rm[s-1]),xm)

        Pu += rpp*np.log(1.0-xp/xip[s-1])/qpp + \
              rpm*np.log(1.0-xm/xim[s-1])/qmp

    return Pu



def Entvisc(x,entold,entnew,fpold,fpnew,D,iV,N,m,h,k,VtoE,c1,c2):
    """ Compute nonlinear viscosity following entropy approach
    """
    nu = np.zeros((m+1,N))
    nuh = np.zeros((1,N))
    onev = np.zeros((m+1,1))

    ente = np.zeros((N+2,2))
    fpe = np.zeros((N+2,2))

    #Compute cell-wise residual by Crank-Nicholson approximation
    Resi = (entnew - entold)/k + (fpnew*np.dot(D,entnew) + \
            fpold*np.dot(D,entold))/h

    #Compute interface jump
    ente = extendDG(entnew[VtoE[0],VtoE[1]], "N", 0, "N", 0)
    fpe = extendDG(fpnew[VtoE[0],VtoE[1]],  "N", 0, "N", 0)

    cL = (fpe[:N,1] + fpe[1:N+1,0])*(ente[:N,1]-ente[1:N+1,0])/2
    cR = (fpe[1:N+1,1] + fpe[2:N+2,0])*(ente[1:N+1,1]-ente[2:N+2,0])/2
    Ji = np.fmax(cL,cR)/h

    Eh = np.dot(iV,entnew)
    NormE = np.max(np.abs(entnew-np.outer(onev,Eh[0,:])),axis=0)

    # Define numerical viscosity
    Di = np.fmax( np.max( np.abs(Resi),axis=0), np.abs(Ji)/NormE )
    nuh = np.fmin( c1*h*np.max( np.abs(fpnew), axis=0), c2*h2**2*Di )

    # Compute continuous viscosity
    nue = np.zeros(N+2)
    nue = np.concatenate(([nuh[0]], nuh, [nuh[N-1]]))

    maxL = np.fmax(nue[:N],nue[1:N+1])
    maxR = np.fmax(nue[1:N+1],nue[2:N+2])

    nu = np.outer(onev, maxL) + \
         (x-np.outer(onev, x[0,:]))/h*(np.outer(onev, maxR-maxL))

    return nu


# Helper, used to compute null space of a matrix
def nullspace(A, atol=1e-13, rtol=0):
 
    A = np.atleast_2d(A)
    u, s, vh = np.linalg.svd(A)
    tol = max(atol, rtol * s[0])
    nnz = (s >= tol).sum()
    ns = vh[nnz:].conj().T
    return ns
