"""
This file contains the specific functions to solve Euler equations
in 1D or 2D using a WENO scheme.
"""

import numpy as np
from Euler import *
from Weno import *
from helpers import extend

## 1D ##
###################################################################################
def EulerWENOrhs1D(x,q,h,k,m,Crec,dw,beta,gamma,maxvel):
    """Evaluate the RHS of Euler equations using a WENO reconstruction"""
    N = len(x)
    dq = np.zeros((3,N))

    qe = np.zeros((3,N+2*m))

    # Extend data and assign boundary conditions for Sod's problem
    xe,qe[0,:] = extend(x, q[0,:], m, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extend(x, q[1,:], m, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extend(x, q[2,:], m, "D", 2.5, "N", 0.0)

    # # Extend data and assign boundary conditions for Sod's problem
    # xe,qe[0,:] = extend(x, q[0,:], m, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extend(x, q[1,:], m, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extend(x, q[2,:], m, "D", 39.166661, "N", 0.0)


    #define cell left and right interface values
    qm = np.zeros((3,N+2))
    qp = np.zeros((3,N+2))

    for i in range(N+2):
        qm[0,i], qp[0,i] = WENO(xe[i:1+(i+2*(m-1))],qe[0,i:1+(i+2*(m-1))],m,Crec,dw,beta)
        qm[1,i], qp[1,i] = WENO(xe[i:1+(i+2*(m-1))],qe[1,i:1+(i+2*(m-1))],m,Crec,dw,beta)
        qm[2,i], qp[2,i] = WENO(xe[i:1+(i+2*(m-1))],qe[2,i:1+(i+2*(m-1))],m,Crec,dw,beta)

    # Change numerical flux here
    dq = - (EulerLF(qp[:,1:N+1], qm[:,2:N+2], gamma, maxvel) - \
            EulerLF(qp[:,:N], qm[:,1:N+1], gamma, maxvel))/h

    return dq

def EulerWENOcharrhs1D(x,q,h,k,m,Crec,dw,beta,gamma,maxvel):
    """Evaluate the RHS of Euler equations using a WENO reconstruction
    on the characteristic variables."""

    N = len(x)
    dq = np.zeros((3,N))

    qe = np.zeros((3,N+2*m))

    # Extend data and assign boundary conditions for Sod's problem
    xe,qe[0,:] = extend(x, q[0,:], m, "D", 1.0, "D", 0.125)
    xe,qe[1,:] = extend(x, q[1,:], m, "D", 0.0, "N", 0.0)
    xe,qe[2,:] = extend(x, q[2,:], m, "D", 2.5, "N", 0.0)

    # # Extend data and assign boundary conditions for Sod's problem
    # xe,qe[0,:] = extend(x, q[0,:], m, "D", 3.857143, "N", 0.0)
    # xe,qe[1,:] = extend(x, q[1,:], m, "D", 10.141852, "D", 0.0)
    # xe,qe[2,:] = extend(x, q[2,:], m, "D", 39.166661, "N", 0.0)


    #define cell left and right interface values
    Rlh = np.zeros(3)
    Rrh = np.zeros(3)
    qm = np.zeros((3,N+2))
    qp = np.zeros((3,N+2))

    for i in range(N+2):
        qloc = qe[:,i:1+(i+2*(m-1))]
        q0 = qloc[:,m-1]
        S,iS,Lam = EulerChar(q0,gamma)
        Rloc = np.dot( iS, qloc )
        Rlh[0], Rrh[0] = WENO(xe[i:1+(i+2*(m-1))],Rloc[0,:],m,Crec,dw,beta)
        Rlh[1], Rrh[1] = WENO(xe[i:1+(i+2*(m-1))],Rloc[1,:],m,Crec,dw,beta)
        Rlh[2], Rrh[2] = WENO(xe[i:1+(i+2*(m-1))],Rloc[2,:],m,Crec,dw,beta)
        qm[:,i] = np.dot(S, Rlh)
        qp[:,i] = np.dot(S, Rrh)

    # Change numerical flux here
    dq = - (EulerLF(qp[:,1:N+1], qm[:,2:N+2], gamma, maxvel) - \
            EulerLF(qp[:,:N], qm[:,1:N+1], gamma, maxvel))/h

    return dq

def EulerWENO1D(x,q,h,m,CFL,gamma,FinalTime):
    """ Integrate 1D Euler equation until FinalTime using an ENO
        scheme and a 3rd order SSP-RK
    """   
    t = 0.0
    tstep = 0

    #Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)

    # Initialize linear weights
    dw = LinearWeights(m,0)

    # Compute smoothness indicator matrices
    beta = np.zeros((m,m,m))
    for r in range(m):
        xl = -1/2 + np.arange(-r,m-r+1)
        beta[:,:,r] = betarcalc(xl,m)


    while t < FinalTime:
        p = (gamma-1)*(q[2,:] - 0.5*(q[1,:]**2)/q[0,:])
        c = np.sqrt(gamma*p/q[0,:])
        maxvel = (c+abs(q[1,:]/q[0,:])).max()
        k = min(FinalTime-t, CFL*h/maxvel)

        #Update solution
        rhsq  = EulerWENOcharrhs1D(x,q,h,k,m,Crec,dw,beta,gamma,maxvel)
        q1 = q + k*rhsq
        rhsq  = EulerWENOcharrhs1D(x,q1,h,k,m,Crec,dw,beta,gamma,maxvel) 
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = EulerWENOcharrhs1D(x,q2,h,k,m,Crec,dw,beta,gamma,maxvel) 
        q  = (q + 2*q2 + 2*k*rhsq)/3

        t = t+k
        tstep += 1
        
    return q

## 2D ##
###################################################################################
def EulerWENOrhs2D(x,y,q,hx,hy,k,m,Crec,dw,beta,gamma):
    """Evaluate right hand side for 2D Euler equation 
        using a WENO method"""

    Nxy = x.shape
    Nx = Nxy[1]
    Ny = Nxy[0]

    dq = np.zeros((4,Ny,Nx))

    # Apply WENO in the x-direction
    qe = np.zeros((4,Nx+2*m))

    for i in range(Ny):
        # Extend data and assign boundary conditions in x-direction
        xe,qe[0,:] = extend(x[i,:],q[0,i,:],m,"N",0,"N",0)
        xe,qe[1,:] = extend(x[i,:],q[1,i,:],m,"N",0,"N",0)
        xe,qe[2,:] = extend(x[i,:],q[2,i,:],m,"N",0,"N",0)
        xe,qe[3,:] = extend(x[i,:],q[3,i,:],m,"N",0,"N",0)

        # define cell left and right interface values
        ql = np.zeros((4,Nx+2))
        qr = np.zeros((4,Nx+2))

        for j in range(Nx+2):
            ql[0,j],qr[0,j] = WENO(xe[j:1+(j+2*(m-1))],qe[0,j:1+(j+2*(m-1))],m,Crec,dw,beta)
            ql[1,j],qr[1,j] = WENO(xe[j:1+(j+2*(m-1))],qe[1,j:1+(j+2*(m-1))],m,Crec,dw,beta)
            ql[2,j],qr[2,j] = WENO(xe[j:1+(j+2*(m-1))],qe[2,j:1+(j+2*(m-1))],m,Crec,dw,beta)
            ql[3,j],qr[3,j] = WENO(xe[j:1+(j+2*(m-1))],qe[3,j:1+(j+2*(m-1))],m,Crec,dw,beta)

        # Compute flux
        dq1 = EulerLF2Dx(qr[:,1:Nx+1],ql[:,2:Nx+2],gamma)
        dq2 = EulerLF2Dx(qr[:,:Nx],ql[:,1:Nx+1],gamma)
        # Update residual
        dq[:,i,:] = - ( dq1-dq2)/hx

    # Apply WENO in the y-direction
    qe = np.zeros((4,Ny+2*m))

    for j in range(Nx):
        # Extend data and assign boundary conditions in x-direction
        ye,qe[0,:] = extend(y[:,j],q[0,:,j],m,"N",0,"N",0)
        ye,qe[1,:] = extend(y[:,j],q[1,:,j],m,"N",0,"N",0)
        ye,qe[2,:] = extend(y[:,j],q[2,:,j],m,"N",0,"N",0)
        ye,qe[3,:] = extend(y[:,j],q[3,:,j],m,"N",0,"N",0)

        # define cell left and right interface values
        ql = np.zeros((4,Ny+2))
        qr = np.zeros((4,Ny+2))

        for i in range(Ny+2):
            ql[0,i],qr[0,i] = WENO(ye[i:1+(i+2*(m-1))],qe[0,i:1+(i+2*(m-1))],m,Crec,dw,beta)
            ql[1,i],qr[1,i] = WENO(ye[i:1+(i+2*(m-1))],qe[1,i:1+(i+2*(m-1))],m,Crec,dw,beta)
            ql[2,i],qr[2,i] = WENO(ye[i:1+(i+2*(m-1))],qe[2,i:1+(i+2*(m-1))],m,Crec,dw,beta)
            ql[3,i],qr[3,i] = WENO(ye[i:1+(i+2*(m-1))],qe[3,i:1+(i+2*(m-1))],m,Crec,dw,beta)

        # Compute flux
        dq1 = EulerLF2Dy(qr[:,1:Ny+1],ql[:,2:Ny+2],gamma)
        dq2 = EulerLF2Dy(qr[:,:Ny],ql[:,1:Ny+1],gamma)
        # Update residual
        dq[:,:,j] -= ( dq1-dq2)/hy

    return dq


def EulerWENO2D(x,y,q,hx,hy,m,gamma,CFL,FinalTime):
    """Integrate 2D Euler equation until FinalTime 
           using a WENO scheme."""

    t = 0.0
    tstep = 0.0
    delta = min(hx,hy)

    #Initialize reconstruction weights
    Crec = np.zeros((m+1,m))
    for r in range(-1,m):
        Crec[r+1,:] = ReconstructWeights(m,r)

    # Initialize linear weights
    dw = LinearWeights(m,0)

    # Compute smoothness indicator matrices
    beta = np.zeros((m,m,m))
    for r in range(m):
        xl = -1/2 + np.arange(-r,m-r+1)
        beta[:,:,r] = betarcalc(xl,m)

    # integrate scheme
    while (t<FinalTime):
        # Set timestep
        p = (gamma-1.0)*(q[3,:,:]-0.5*(q[1,:,:]**2+q[2,:,:]**2)/q[0,:,:])
        c = np.sqrt(gamma*p/q[0,:,:])
        maxvelu = (c+np.abs(q[1,:,:]/q[0,:,:])).max()
        maxvelv = (c+np.abs(q[2,:,:]/q[0,:,:])).max()
        k = min(FinalTime-t, CFL*delta/np.sqrt(maxvelu**2+maxvelv**2)/np.sqrt(2))
  
        # Update solution
        rhsq  = EulerWENOrhs2D(x,y,q,hx,hy,k,m,Crec,dw,beta,gamma)
        q1 = q + k*rhsq
        rhsq  = EulerWENOrhs2D(x,y,q1,hx,hy,k,m,Crec,dw,beta,gamma)
        q2 = (3*q + q1 + k*rhsq)/4
        rhsq  = EulerWENOrhs2D(x,y,q2,hx,hy,k,m,Crec,dw,beta,gamma)
        q = (q + 2*q2 + 2*k*rhsq)/3

        t += k
        tstep += 1

    return q

