"""
This file contains the specific functions to solve Euler equations
in 1D using a spectral approach.
"""

import numpy as np
from Euler import *
from Spectral import *


def EulerSpecrhs1D(x,q,L,gamma):
    """Purpose  : Evaluate the RHS of Euler equations using spectral Fourier
                  collocation method"""

    N = len(x)
    r = q[0,:]
    ru = q[1,:]
    E = q[2,:]
    p = (gamma-1.0)*(E-0.5*ru**2/r)

    dq = np.zeros((3,N))

    dq[0,:] = -2.0*np.pi/L*Fourierdx(ru,1.0)
    dq[1,:] = -2.0*np.pi/L*Fourierdx(ru**2/r+p,1.0)
    dq[2,:] = -2.0*np.pi/L*Fourierdx((E+p)*ru/r,1.0)

    return dq

def EulerSpec1D(x,q,N,L,CFL,gamma,FinalTime):
    """Purpose  : Integrate 1D Euler equation until FinalTime using a Fourier
                  spectral collocation method and 3rd order SSP-RK method
    """   

    t = 0.0
    tstep = 0
    h = L/(2.0*N+1)

    # Parameter for hyper viscosity
    p = 8.0

    #integrate scheme
    while (t<FinalTime):
        # Decide on timestep
        pres = (gamma-1.0)*(q[2,:]-0.5*q[1,:]**2/q[0,:])
        c = np.sqrt(gamma*pres/q[0,:])
        maxvel = (c + np.abs(q[1,:]/q[0,:])).max()
        k = min(CFL*h/maxvel,FinalTime-t)
        # Update solution 
        rhsq  = EulerSpecrhs1D(x,q,L,gamma)
        q1 = q + k*rhsq
        q1[0,:] = FourierVanishHypVisc(q1[0,:],p,k,N,L)
        q1[1,:] = FourierVanishHypVisc(q1[1,:],p,k,N,L)
        q1[2,:] = FourierVanishHypVisc(q1[2,:],p,k,N,L)

        rhsq  = EulerSpecrhs1D(x,q1,L,gamma)
        q2 = (3*q + q1 + k*rhsq)/4
        q2[0,:] = FourierVanishHypVisc(q2[0,:],p,k,N,L)
        q2[1,:] = FourierVanishHypVisc(q2[1,:],p,k,N,L)
        q2[2,:] = FourierVanishHypVisc(q2[2,:],p,k,N,L)

        rhsq  = EulerSpecrhs1D(x,q2,L,gamma)
        q = (q + 2*q2 + 2*k*rhsq)/3
        q[0,:] = FourierVanishHypVisc(q[0,:],p,k,N,L)
        q[1,:] = FourierVanishHypVisc(q[1,:],p,k,N,L)
        q[2,:] = FourierVanishHypVisc(q[2,:],p,k,N,L)
        
        t += k
        tstep += 1

    return q
