"""
This file contains all the basic implementations of functions to compute
quantities for the specific case of the Euler equations.
"""
import numpy as np
import tinyarray as ta

## 1D ##
###################################################################################
def EulerFlux(q, gamma):
    """Purpose: Compute flux for 1D Euler equations."""

    r = q[0,:]
    ru = q[1,:]
    E = q[2,:]
    p = (gamma-1)*(E - 0.5*ru**2/r);

    flux = np.array((ru, ru**2/r + p, (E + p)*ru/r))

    return flux

def EulerJac(q, gamma):
    """Purpose: Compute Jacobian for 1D Euler equations. """

    r = q[0]
    ru = q[1]
    E = q[2]

    u = ru/r

    return ta.array(((0.0,1.0,0.0), \
                     (-0.5*(3.0-gamma)*(u**2), (3.0-gamma)*u, gamma-1.0 ), \
                     (-gamma*u*E/r + (gamma-1.0)*(u**3), gamma*E/r - 1.5*(gamma-1.0)*(u**2), gamma*u  )))


def EulerLF(u, v, gamma, maxvel):
    """Purpose: Evaluate global Lax Friedrich numerical flux for 
                the Euler equations
	"""
    r = u[0,:]
    ru = u[1,:]
    E = u[2,:]
    pu = (gamma-1)*(E - 0.5*ru**2/r);
    
    fu = np.array((ru, ru**2/r+pu, (E+pu)*ru/r))
    
    r = v[0,:]
    ru = v[1,:]
    E = v[2,:]
    pv = (gamma-1)*(E - 0.5*ru**2/r);
    
    fv = np.array((ru, ru**2/r+pv, (E+pv)*ru/r))
    
    flux = 0.5*(fu+fv)-0.5*maxvel*(v-u)
    return flux

def EulerLLF(u, v, gamma):
    """Purpose: Evaluate local Lax Friedrich numerical flux for 
                the Euler equations
    """

    # Compute flux for u
    r = u[0,:]
    ru = u[1,:]
    E = u[2,:]
    pu = (gamma-1)*(E - 0.5*ru**2/r)

    cu = np.sqrt(gamma*pu/r)
    
    fu = np.array((ru, ru**2/r+pu, (E+pu)*ru/r))
    
    # Compute flux for v
    r = v[0,:]
    ru = v[1,:]
    E = v[2,:]
    pv = (gamma-1)*(E - 0.5*ru**2/r)

    cv = np.sqrt(gamma*pv/r)
    
    fv = np.array((ru, ru**2/r+pv, (E+pv)*ru/r))

    # Evaluate numerical flux
    localvel = np.fmax( cu + np.abs(u[1,:]/u[0,:]), \
                        cv + np.abs(v[1,:]/v[0,:]) )
    
    numflux = 0.5*(fu+fv)-0.5*np.outer(np.ones(3), localvel)*(v-u)
    return flux

def EulerLW(u,v,gamma,lambd):
    """Purpose: Evaluate Lax Wendroff numerical flux for the Euler equations"""

    # Compute flux for u
    r = u[0,:]
    ru = u[1,:]
    E = u[2,:]
    pu = (gamma-1)*(E - 0.5*ru**2/r);
    
    fu = np.array((ru, ru**2/r+pu, (E+pu)*ru/r))
    
    # Compute flux for v
    r = v[0,:]
    ru = v[1,:]
    E = v[2,:]
    pv = (gamma-1)*(E - 0.5*ru**2/r);
    
    fv = np.array((ru, ru**2/r+pv, (E+pv)*ru/r))

    # Evaluate numerical flux
    fw = fv-fu
    w = 0.5*(u+v)
    rw = w[0,:]
    ruw = w[1,:]
    Ew = w[2,:]
    uw = ruw/rw
    wL = len(rw)

    A21 = -(3.0-gamma)/2*uw**2
    A22 = (3.0-gamma)*uw
    A23 = (gamma-1.0)*np.ones(wL)

    A31 = -gamma*Ew*uw/rw + (gamma-1.0)*uw**3
    A32 = gamma*Ew/rw - 3*(gamma-1.0)/2*uw**2
    A33 = gamma*uw

    LFvec = np.zeros((3,wL))
    LFvec[0,:] = fw[1,:] 
    LFvec[1,:] = A21*fw[0,:] + A22*fw[1,:] + A23*fw[2,:]
    LFvec[2,:] = A31*fw[0,:] + A32*fw[1,:] + A33*fw[2,:]

    numflux = 0.5*((fu+fv) - lambd*LFvec)
    return numflux

def EulerChar(q0,gamma):
    """Purpose: Compute characteristic decomposition for Euler equations at q0"""

    r0 = q0[0]
    ru0 = q0[1]
    E0 = q0[2]

    u0 = ru0/r0
    p0 = (gamma-1.0)*(E0-0.5*ru0**2/r0)
    c0 = np.sqrt(gamma*p0/r0)
    H0 = (E0+p0)/r0

    S = np.zeros((3,3))
    iS = np.zeros((3,3))
    Lam = np.zeros((3,3))

    S[0,0] = 1.0/(2.0*c0)
    S[0,1] = 1.0
    S[0,2] = 1.0/(2.0*c0)

    S[1,0] = (u0+c0)/(2.0*c0)
    S[1,1] = u0
    S[1,2] = (u0-c0)/(2.0*c0)

    S[2,0] = (H0+c0*u0)/(2.0*c0)
    S[2,1] = 0.5*u0**2
    S[2,2] = (H0-c0*u0)/(2.0*c0)


    iS[0,0] = (0.5*(gamma-1.0)*u0**2-c0*u0)/c0
    iS[0,1] = -((gamma-1.0)*u0-c0)/c0
    iS[0,2] = (gamma-1.0)/c0

    iS[1,0] = 1.0-0.5*(gamma-1.0)*u0**2/c0**2
    iS[1,1] = (gamma-1.0)/c0**2*u0
    iS[1,2] = -(gamma-1)/c0**2

    iS[2,0] = (0.5*(gamma-1.0)*u0**2+c0*u0)/c0
    iS[2,1] = -((gamma-1.0)*u0+c0)/c0
    iS[2,2] = (gamma-1.0)/c0

    Lam[0,0] = u0+c0
    Lam[1,1] = u0
    Lam[2,2] = u0-c0

    return S,iS,Lam


def EulerRoe(u,v,gamma):
    """Purpose: Evaluate Roe numerical flux for Euler's equation. 
                No entropy fix"""

    # Compute flux for u
    ru = u[0,:]
    ruu = u[1,:]
    Eu = u[2,:]

    uu = ruu/ru
    pu = (gamma-1.0)*(Eu-0.5*ruu**2/ru)
    Hu = (Eu+pu)/ru

    fu = np.array(( ruu, ruu**2/ru+pu, Hu*ruu ))

    # Compute flux for v
    rv = v[0,:]
    ruv = v[1,:]
    Ev = v[2,:]

    uv = ruv/rv
    pv = (gamma-1.0)*(Ev-0.5*ruv**2/rv)
    Hv = (Ev+pv)/rv

    fv = np.array(( ruv, ruv**2/rv+pv, Hv*ruv ))

    # Compute Roe averaged variables
    uh = (np.sqrt(ru)*uu+np.sqrt(rv)*uv)/(np.sqrt(ru)+np.sqrt(rv))
    Hh = (np.sqrt(ru)*Hu+np.sqrt(rv)*Hv)/(np.sqrt(ru)+np.sqrt(rv))
    ch = np.sqrt((gamma-1.0)*(Hh-0.5*uh**2))

    a2 = (gamma-1.0)*((rv-ru)*(Hh-uh**2) + uh*(ruv-ruu) - (Ev-Eu))/ch**2
    a1 = ((rv-ru)*(uh+ch) - (ruv-ruu) - ch*a2)/(2.0*ch)
    a3 = (rv-ru) - (a1+a2)

    Unit = np.ones(len(uh))

    s1 = np.array((Unit, uh-ch, Hh-uh*ch))
    s2 = np.array((Unit, uh, 0.5*uh**2))
    s3 = np.array((Unit, uh+ch, Hh+uh*ch))

    numflux = 0.5*(fu+fv) - 0.5*(s1*(a1*(np.abs(uh-ch)[None,:])) + \
                                 s2*(a2*(np.abs(uh)[None,:])) + \
                                 s3*(a3*(np.abs(uh+ch)[None,:])))
    return numflux

## 2D ##
###################################################################################
def EulerFlux2Dx(q, gamma):
    """Purpose: Evaluate x-flux for 2D Euler equations"""
    r = q[0,:]
    ru = q[1,:]
    rv = q[2,:]
    E = q[3,:]

    u = ru/r
    v = rv/r

    p = (gamma-1)*(E - 0.5*r*(u**2+v**2))

    return np.array(( ru, ru*u+p, ru*v, (E+p)*u ))

def EulerFlux2Dy(q, gamma):
    """Purpose: Evaluate y-flux for 2D Euler equations"""
    r = q[0,:]
    ru = q[1,:]
    rv = q[2,:]
    E = q[3,:]

    u = ru/r
    v = rv/r

    p = (gamma-1)*(E - 0.5*r*(u**2+v**2))

    return np.array(( rv, rv*u, rv*v+p, (E+p)*v ))

def EulerJac2Dx(q, gamma):

    """Purpose: Evaluate x-Jacobian for 2D Euler equations"""

    r = q[0]
    ru = q[1]
    rv = q[2]
    E = q[3]

    u = ru/r
    v = rv/r

    p = (gamma-1)*(E - 0.5*(ru**2 + rv**2)/r)
    pE = 0.5*(gamma-1)*(u**2+v**2)

    return ta.array(( (0.0, 1.0, 0.0, 0.0), \
                      (pE-u**2, (3-gamma)*u, -(gamma-1)*v, gamma-1), \
                      (-u*v, v, u, 0.0), \
                      ( (2*pE-gamma*E/r)*u, gamma*E/r-pE-(gamma-1)*(u**2), -(gamma-1)*u*v, gamma*u ) ))


def EulerJac2Dy(q, gamma):

    """Purpose: Evaluate y-Jacobian for 2D Euler equations"""

    r = q[0]
    ru = q[1]
    rv = q[2]
    E = q[3]

    u = ru/r
    v = rv/r

    p = (gamma-1)*(E - 0.5*(ru**2 + rv**2)/r)
    pE = 0.5*(gamma-1)*(u**2+v**2)

    return ta.array(( (0.0, 0.0, 1.0, 0.0), \
                      (-u*v, v, u, 0.0), \
                      (pE-v**2, -(1-gamma)*u, (3-gamma)*v,  gamma-1), \
                      ( (2*pE-gamma*E/r)*v, -(gamma-1)*u*v, gamma*E/r-pE-(gamma-1)*(v**2), gamma*v ) ))




def EulerLF2Dx(u, v, gamma):
    """Purpose: Evaluate Lax Friedrich numerical flux for Euler's equation along x"""
 
    # Extract states
    rl = u[0,:]
    rul = u[1,:]
    rvl = u[2,:]
    El = u[3,:]

    rr = v[0,:]
    rur = v[1,:]
    rvr = v[2,:]
    Er = v[3,:]

    # Compute fluxes for left and right states
    ul = rul/rl
    vl = rvl/rl
    pl = (gamma-1.0)*(El - 0.5*rl*(ul**2+vl**2))
    cl = np.sqrt(gamma*pl/rl)

    fl = np.array(( rul, rul*ul + pl, rul*vl, (El+pl)*ul ))

    ur = rur/rr
    vr = rvr/rr
    pr = (gamma-1.0)*(Er - 0.5*rr*(ur**2+vr**2))
    cr = np.sqrt(gamma*pr/rr)

    fr = np.array(( rur, rur*ur + pr, rur*vr, (Er+pr)*ur ))

    # Compute dissipation for LF
    maxvell = cl + np.abs(ul)
    maxvelr = cr + np.abs(ur)
    alpha = (np.fmax(maxvell,maxvelr)).max() # Global LF
    # alpha = np.outer(np.ones(4), np.fmax(maxvell,maxvelr)) # Local LF
    
    return 0.5*(fl+fr)-0.5*alpha*(v-u)

 
def EulerLF2Dy(u, v, gamma):
    """Purpose: Evaluate Lax Friedrich numerical flux for Euler's equation along y"""   
    
    # Extract states
    rl = u[0,:]
    rul = u[1,:]
    rvl = u[2,:]
    El = u[3,:]

    rr = v[0,:]
    rur = v[1,:]
    rvr = v[2,:]
    Er = v[3,:]

    # Compute fluxes for left and right states
    ul = rul/rl
    vl = rvl/rl
    pl = (gamma-1.0)*(El - 0.5*rl*(ul**2+vl**2))
    cl = np.sqrt(gamma*pl/rl)

    fl = np.array(( rvl, rvl*ul, rvl*vl+pl, (El+pl)*vl ))

    ur = rur/rr
    vr = rvr/rr
    pr = (gamma-1.0)*(Er - 0.5*rr*(ur**2+vr**2))
    cr = np.sqrt(gamma*pr/rr)

    fr = np.array(( rvr, rvr*ur, rvr*vr+pr, (Er+pr)*vr ))

    # Compute dissipation for LF
    maxvell = cl + np.abs(vl)
    maxvelr = cr + np.abs(vr)
    alpha = (np.fmax(maxvell,maxvelr)).max() # Global LF
    # alpha = np.outer(np.ones(4), np.fmax(maxvell,maxvelr)) # Local LF
    
    return 0.5*(fl+fr)-0.5*alpha*(v-u)
    

