"""
This file contains the specific functions to solve Burgers equations
in 1D or 2D using a slope limited scheme.
"""

import numpy as np
from Burgers import *
from helpers import extend,SlopeLimit

## 1D ##
###################################################################################
def BurgersSLrhs1D(x, u, h, k, maxvel):

    """Purpose: Evaluate right hand side for Burgers equation using a
                slope limited scheme"""
    N = len(x)

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 2
    c = 0.0
    M = 10.0

    # Boundary conditions 
    xe, ue = extend(x, u, 2, "P", 0, "P", 0) # Periodic boundary conditions
    #xe, ue = extend(x, u, 2, "D", 2, "N", 0) # Constant boundary conditions

    # Compute element slope and limit
    dup = ue[2:N+4]-ue[1:N+3]
    dum = ue[1:N+3]-ue[:N+2]
    duL = SlopeLimit(dup,dum,typ,c,M,h)

    # Compute cell interface values - for f'(u) = 2u
    uloc = ue[1:N+3]
    uh = uloc - 2*uloc*k/(2.0*h)*duL
    uL = uh - 0.5*duL
    uR = uh + 0.5*duL


    # Compute RHS
    du = - (BurgersLF(uR[1:N+1], uL[2:N+2], maxvel) - \
            BurgersLF(uR[:N], uL[1:N+1],maxvel))/h
    return du

def BurgersSL1D(x, u, h, CFL, FinalTime):
    """Purpose  : Integrate 1D Burgers equation until FinalTime 
                using a slope limited scheme.
    """   
    t = 0.0
    timestep = 0

    # Integrate scheme
    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.abs(u)).max()
        k = min(FinalTime-t, CFL*h/maxvel)
        # Update solution
        u += k*BurgersSLrhs1D(x,u,h,k,maxvel)
        t +=k
        timestep += 1
        
    return u

## 2D ##
###################################################################################
def BurgersSLrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D Burgers equation 
                using a slope limited scheme
    """   

    Ny,Nx = u.shape

    du = np.zeros((Ny,Nx))

    # Chose slope limiter - 0:LF; 1:minmod; 2:MUSCL; 3:Superbee; 
    # 4:van Albada; 5:van Leer, 6: TVB 
    typ = 5 
    c = 0.0
    M = 10.0

    # Extend data and assign boundary conditions
    for i in range(Ny):
        xe,ue = extend(x[i,:], u[i,:], 2, "P", 0, "P", 0)

        # Compute element slope and limit
        dup = ue[2:Nx+4]-ue[1:Nx+3]
        dum = ue[1:Nx+3]-ue[:Nx+2]
        duL = SlopeLimit(dup,dum,typ,c,M,hx)

        # Compute cell interface values 
        uloc = ue[1:Nx+3]
        uh = uloc - BurgersJac2Dx(uloc)*k/(2.0*hx)*duL
        uL = uh - 0.5*duL
        uR = uh + 0.5*duL

        # Compute RHS
        du[i,:] = - (BurgersLF(uR[1:Nx+1], uL[2:Nx+2], maxvel) - \
                BurgersLF(uR[:Nx], uL[1:Nx+1],maxvel))/hx


    for j in range(Nx):
        xe,ue = extend(y[:,j], u[:,j], 2, "P", 0, "P", 0)

        # Compute element slope and limit
        dup = ue[2:Ny+4]-ue[1:Ny+3]
        dum = ue[1:Ny+3]-ue[:Ny+2]
        duL = SlopeLimit(dup,dum,typ,c,M,hy)

        # Compute cell interface values 
        uloc = ue[1:Ny+3]
        uh = uloc - BurgersJac2Dy(uloc)*k/(2.0*hy)*duL
        uL = uh - 0.5*duL
        uR = uh + 0.5*duL

        # Compute RHS
        du[:,j] -= (BurgersLF(uR[1:Ny+1], uL[2:Ny+2], maxvel) - \
                BurgersLF(uR[:Ny], uL[1:Ny+1],maxvel))/hy
        
    return du
    

def BurgersSL2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose  : Integrate 2D Burgers equation until FinalTime 
                using a slope limited scheme.
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    
    # Integrate scheme
    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.sqrt(2.0)*np.abs(u)).max()
        k = min(FinalTime-t, CFL*delta/maxvel/2.0)
        # Update solution
        u += k*BurgersSLrhs2D(x, y, u, hx, hy, k, maxvel)
        t += k
        tstep += 1
        
    return u