"""
This file contains the specific functions to solve Burgers equations
in 1D or 2D using a flux-limited scheme.
"""

import numpy as np
from Burgers import *
from helpers import extend,FluxLimit

## 1D ##
###################################################################################
def BurgersFLrhs1D(x, u, h, k, maxvel):

    """Purpose: Evaluate right hand side for Burgers equation using a
                flux limited scheme"""
    N = len(x)

    # Chose flux limiter - 0:LF; 1:CO; 2:Koren; 3:Sweby; 4:OSPRE; 5:van Leer
    typ = 5
    beta = 2.0

    # Boundary conditions 
    xe, ue = extend(x, u, 1, "P", 0, "P", 0); # Periodic boundary conditions
    #xe, ue = extend(x, u, 1, "D", 2, "N", 0); % Constant boundary conditions

    # Compute indicator function and define flux limiter
    r = (ue[1:N+1] - ue[:N])/(ue[2:N+2] - ue[1:N+1])
    xe,re = extend(x,r,1,"N",0,"N",0)
    rm = 1.0/re
    phiLp = FluxLimit( re[:N], typ, beta )
    phiRp = FluxLimit( re[1:N+1], typ, beta )
    phiLm = FluxLimit( rm[1:N+1], typ, beta )
    phiRm = FluxLimit( rm[2:N+2], typ, beta )

    ufilt = u >= 0.0
    phiL = ufilt*phiLp + (1-ufilt)*phiLm
    phiR = ufilt*phiRp + (1-ufilt)*phiRm

    # Compute left flux - Change numerical flux here
    Fluxlow = BurgersLF(ue[:N],ue[1:N+1],maxvel)
    Fluxhigh = BurgersLW( ue[:N],ue[1:N+1],k/h)
    FluxL = Fluxlow + phiL*(Fluxhigh-Fluxlow)

    # Compute right flux - Change numerical flux here
    Fluxlow = BurgersLF(ue[1:N+1],ue[2:N+2],maxvel)
    Fluxhigh = BurgersLW( ue[1:N+1],ue[2:N+2],k/h)
    FluxR = Fluxlow + phiR*(Fluxhigh-Fluxlow)

    # Compute RHS
    du = - (FluxR-FluxL)/h
    return du

def BurgersFL1D(x, u, h, CFL, FinalTime):
    """Purpose  : Integrate 1D Burgers equation until FinalTime using a flux limited scheme.
    """   
    t = 0.0
    timestep = 0

    while t < FinalTime:
        maxvel = (2*np.abs(u)).max()
        k = min(FinalTime-t, CFL*h/maxvel)
        u += k*BurgersFLrhs1D(x,u,h,k,maxvel)
        t +=k
        timestep += 1
        
    return u

## 2D ##
###################################################################################
def BurgersFLrhs2D(x, y, u, hx, hy, k, maxvel):
    """Purpose: Evaluate right hand side for 2D Burgers equation 
                using a flux limited scheme
    """   

    Ny,Nx = u.shape

    du = np.zeros((Ny,Nx))

    # Chose flux limiter - 0:LF; 1:CO; 2:Koren; 3:Sweby; 4:OSPRE; 5:van Leer 
    typ = 5
    beta = 1.5

    # Extend data and assign boundary conditions
    for i in range(Ny):
        xe,ue = extend(x[i,:], u[i,:], 1, "P", 0, "P", 0)

        # Compute indicator function and define flux limiter
        r = (ue[1:Nx+1]-ue[:Nx])/(ue[2:Nx+2]-ue[1:Nx+1])
        xe,re = extend(x[i,:],r,1,"N",0,"N",0)
        rm = 1.0/re
        phiLp = FluxLimit( re[:Nx],typ,beta )
        phiRp = FluxLimit( re[1:Nx+1],typ,beta )
        phiLm = FluxLimit( rm[1:Nx+1],typ,beta )
        phiRm = FluxLimit( rm[2:Nx+2],typ,beta )

        ufilt = u[i,:] >= 0.0
        phiL = ufilt*phiLp + (1-ufilt)*phiLm
        phiR = ufilt*phiRp + (1-ufilt)*phiRm

        # Compute left flux - Change numerical flux here
        Fluxlow = BurgersLF( ue[:Nx],ue[1:Nx+1],maxvel )
        Fluxhigh = BurgersLW( ue[:Nx], ue[1:Nx+1], k/hx)
        FluxL = Fluxlow - phiL*(Fluxlow-Fluxhigh)

        # Compute right flux - Change numerical flux here
        Fluxlow = BurgersLF( ue[1:Nx+1],ue[2:Nx+2],maxvel )
        Fluxhigh = BurgersLW( ue[1:Nx+1], ue[2:Nx+2], k/hx)
        FluxR = Fluxlow - phiR*(Fluxlow-Fluxhigh)

        # Update residual
        du[i,:] = - (FluxR-FluxL)/hx

    for j in range(Nx):
        xe,ue = extend(y[:,j], u[:,j], 1, "P", 0, "P", 0)

        # Compute indicator function and define flux limiter
        r = (ue[1:Ny+1]-ue[:Ny])/(ue[2:Ny+2]-ue[1:Ny+1])
        xe,re = extend(y[:,j],r,1,"N",0,"N",0)
        rm = 1.0/re
        phiLp = FluxLimit( re[:Ny],typ,beta )
        phiRp = FluxLimit( re[1:Ny+1],typ,beta )
        phiLm = FluxLimit( rm[1:Ny+1],typ,beta )
        phiRm = FluxLimit( rm[2:Ny+2],typ,beta )

        ufilt = u[:,j] >= 0.0
        phiL = ufilt*phiLp + (1-ufilt)*phiLm
        phiR = ufilt*phiRp + (1-ufilt)*phiRm

        # Compute left flux - Change numerical flux here
        Fluxlow = BurgersLF( ue[:Ny],ue[1:Ny+1],maxvel )
        Fluxhigh = BurgersLW( ue[:Ny], ue[1:Ny+1], k/hy)
        FluxL = Fluxlow - phiL*(Fluxlow-Fluxhigh)

        # Compute right flux - Change numerical flux here
        Fluxlow = BurgersLF( ue[1:Ny+1],ue[2:Ny+2],maxvel )
        Fluxhigh = BurgersLW( ue[1:Ny+1], ue[2:Ny+2], k/hy)
        FluxR = Fluxlow - phiR*(Fluxlow-Fluxhigh)

        # Update residual
        du[:,j] -= (FluxR-FluxL)/hy
        
    return du
    

def BurgersFL2D(x, y, u, hx, hy, CFL, FinalTime):
    """Purpose  : Integrate 2D Burgers equation until FinalTime using a flux limited scheme.
    """   
    t = 0.0
    tstep = 0
    delta = min(hx,hy)
    
    while t < FinalTime:
        # Decide on timestep
        maxvel = (2*np.sqrt(2.0)*np.abs(u)).max()
        k = min(FinalTime-t, CFL*delta/maxvel/2.0)
        # Update solution
        u += k*BurgersFLrhs2D(x, y, u, hx, hy, k, maxvel)
        t += k
        tstep += 1
        
    return u