from util import np, _
from parabolic import implicit_euler, make_video, forward_euler
from integrate import assemble_matrix_from_iterables, \
                      mass_with_reaction_iter, \
                      stiffness_with_diffusivity_iter, \
                      assemble_rhs_from_iterables, \
                      poisson_rhs_iter
from mesh import Triangulation
from quad import seven_point_gauss_6


def main():
  mesh = Triangulation.from_polygon(np.array([ [0, 0],
                                               [1, 0],
                                               [1, 1],
                                               [0, 1] ]), mesh_size=0.04)
  quadrule = seven_point_gauss_6()
  M = assemble_matrix_from_iterables(mesh, mass_with_reaction_iter(mesh, quadrule))
  S = -assemble_matrix_from_iterables(mesh, stiffness_with_diffusivity_iter(mesh, quadrule))

  bindices = mesh.boundary_indices
  data = np.zeros(bindices.shape)

  x, y = mesh.points.T
  lhs0 = 0.5 * np.sin(np.pi*x) * np.sin(np.pi * y)
  timestep = 0.05
  T = 10

  f0 = 50 * assemble_rhs_from_iterables(mesh, poisson_rhs_iter(mesh, quadrule, lambda x: np.sin(2 * np.pi * x[:, 0]) * np.sin(2 * np.pi * x[:, 1])))
  f1 = 100 * assemble_rhs_from_iterables(mesh, poisson_rhs_iter(mesh, quadrule, lambda x: np.sin(4 * np.pi * x[:, 0]) * np.sin(4 * np.pi * x[:, 1])))

  ft = lambda t: np.sin(2 * np. pi * t + np.pi / 2) * ((1 - t/T) * f0 + t / T * f1)

  solutions, time_instances = implicit_euler(M, S, timestep, lhs0, T, ft=ft,
                                                                      freezeindices=bindices,
                                                                      data=data,
                                                                      solverargs={'method': 'cg'})
  make_video(mesh, solutions, time_instances, filename='euler.mp4')


if __name__ == '__main__':
  main()
