from util import np
from quad import QuadRule, seven_point_gauss_6
from integrate import mass_with_reaction_iter, stiffness_with_diffusivity_iter, \
                      assemble_matrix_from_iterables, transport_matrix_iter, \
                      transport_matrix_with_stabilisation_iter, poisson_rhs_iter, \
                      assemble_rhs_from_iterables, streamline_diffusion_stabilisation_iter
from mesh import Triangulation
from solve import solve_with_dirichlet_data


"""
  This example teaches you how to use the code to solve problems subject to
  Dirichlet boundary conditions.
"""
def homogeneous_poisson_with_sine_rhs(quadrule: QuadRule, mesh_size=0.025, k=1):
  """
    P1 FEM solution of the poisson-type problem

      -∆u = sin(2 k π x) sin(2 k π y)    in  Ω
        u = 0                            on ∂Ω

    where Ω = (0, 1)^2.

    parameters
    ----------
    quadrule: `QuadRule`
      quadrature scheme used to assemble the system matrices and right-hand-side.
    mesh_size: float value 0 < mesh_size < 1 tuning the mesh density.
               Smaller value => denser mesh.
    k: `int`
      the k-value of the right-hand-side
  """

  assert isinstance(k, (int, np.int_))

  square = np.array([ [0, 0],
                      [1, 0],
                      [1, 1],
                      [0, 1] ])
  mesh = Triangulation.from_polygon(square, mesh_size=mesh_size)

  # all boundary indices
  bindices = mesh.boundary_indices

  # the data is zero on the whole boundary
  data = np.zeros(bindices.shape, dtype=float)

  # assemble the stiffness matrix
  Aiter = stiffness_with_diffusivity_iter(mesh, quadrule)
  A = assemble_matrix_from_iterables(mesh, Aiter)

  # assemble the right-hand-side vector
  rhs_iter = poisson_rhs_iter(mesh, quadrule, lambda x: np.sin(2 * np.pi * k * x[:, 0]) * np.sin(2 * np.pi * k * x[:, 1]))
  rhs = assemble_rhs_from_iterables(mesh, rhs_iter)

  # solve with Dirichlet data
  solution = solve_with_dirichlet_data(A, rhs, bindices, data)

  # plot
  mesh.tripcolor(solution, title=r'The numerical solution of $-\Delta u = \sin(2 k \pi x) \sin(2 k \pi y)$ with $k = {}$'.format(k))


def session07_exercise_01_no_stab(quadrule: QuadRule,
                                  mesh_size=0.025,
                                  f: float = 0):
  """
    P1 FEM solution of the convection-diffusion-reaction equation

      -∆u + div(beta u) + u = f    in  Ω
                          u = 1    on the bottom and left part of ∂Ω
                          u = 0    else

    where Ω = (0, 1)^2.

    parameters
    ----------
    quadrule: `QuadRule`
      quadrature scheme used to assemble the system matrices and right-hand-side.
    mesh_size: float value 0 < mesh_size < 1 tuning the mesh density.
               Smaller value => denser mesh.
    f: float value representing a constant right hand side `f`. Defaults to 0.
  """
  square = np.array([ [0, 0],
                      [1, 0],
                      [1, 1],
                      [0, 1] ])
  mesh = Triangulation.from_polygon(square, mesh_size=mesh_size)

  # all boundary indices
  bindices = mesh.boundary_indices

  # corresponding boundary points
  bpoints = mesh.points[bindices]

  # data array of same length containing (initially) only zeros
  data = np.zeros(bindices.shape, dtype=float)

  # set data array to one on the left boundary
  # on the left boundary the x-coordinate of the points is machine precision 0
  left_mask = np.abs(bpoints[:, 0]) < 1e-13
  data[left_mask] = 1

  # set data array to one on the bottom boundary
  # on the bottom boundary the y-coordinate of the points is machine precision 0
  bottom_mask = np.abs(bpoints[:, 1]) < 1e-13
  data[bottom_mask] = 1

  beta = lambda x: np.array([[-1e3, -1e3]])

  Miter = mass_with_reaction_iter(mesh, quadrule)
  Aiter = stiffness_with_diffusivity_iter(mesh, quadrule)
  Biter = transport_matrix_iter(mesh, quadrule, beta)

  S = assemble_matrix_from_iterables(mesh, Miter, Aiter, Biter)

  ffunc = lambda x: np.array([f])
  rhs = assemble_rhs_from_iterables(mesh, poisson_rhs_iter(mesh, quadrule, f=ffunc))

  solution = solve_with_dirichlet_data(S, rhs, bindices, data)

  mesh.tripcolor(solution, title='The solution for h={:.3g} without stabilisation'.format(mesh_size))


def session07_exercise01_streamline(quadrule: QuadRule,
                                    mesh_size=0.025,
                                    gamma=5,
                                    f: float = 0):
  """
    P1 FEM solution of the convection-diffusion-reaction equation with streamline diffusion stabilisation

      -∆u + div(beta u) + u = f    in  Ω
                          u = 1    on the bottom and left part of ∂Ω
                          u = 0    else

    where Ω = (0, 1)^2.

    parameters
    ----------
    quadrule: `QuadRule`
      quadrature scheme used to assemble the system matrices and right-hand-side.
    mesh_size: float value 0 < mesh_size < 1 tuning the mesh density.
               Smaller value => denser mesh.
    gamma: `float`
      Stabilisation factor. Gamma bigger => more artificial diffusion.
  """

  square = np.array([ [0, 0],
                      [1, 0],
                      [1, 1],
                      [0, 1] ])
  mesh = Triangulation.from_polygon(square, mesh_size=mesh_size)

  # all boundary indices
  bindices = mesh.boundary_indices

  # corresponding boundary points
  bpoints = mesh.points[bindices]

  # data array of same length containing (initially) only zeros
  data = np.zeros(bindices.shape, dtype=float)

  # set data array to one on the left boundary
  # on the left boundary the x-coordinate of the points is machine precision 0
  left_mask = np.abs(bpoints[:, 0]) < 1e-13
  data[left_mask] = 1

  # set data array to one on the bottom boundary
  # on the bottom boundary the y-coordinate of the points is machine precision 0
  bottom_mask = np.abs(bpoints[:, 1]) < 1e-13
  data[bottom_mask] = 1

  beta = lambda x: np.array([[-1e3, -1e3]])

  Miter = mass_with_reaction_iter(mesh, quadrule)
  Aiter = stiffness_with_diffusivity_iter(mesh, quadrule)
  Biter = transport_matrix_iter(mesh, quadrule, beta)
  Stabiter =  # ???

  S = assemble_matrix_from_iterables(mesh, Miter, Aiter, Biter, Stabiter)

  ffunc = lambda x: np.array([f])
  rhs = assemble_rhs_from_iterables(mesh, poisson_rhs_iter(mesh, quadrule, f=ffunc))

  solution = solve_with_dirichlet_data(S, rhs, bindices, data)

  mesh.tripcolor(solution, title='The solution with streamline diffusion stabilisation for h={:.3g} and gamma={:.3g}'.format(mesh_size, gamma))


def solve_2():

  quadrule = seven_point_gauss_6()

  for mesh_size in (0.1, 0.05, 0.025):
    session07_exercise_01_no_stab(quadrule, mesh_size, f=0)
    for gamma in (0.1, 1, 5):
      session07_exercise01_streamline(quadrule, mesh_size, gamma=gamma, f=0)



if __name__ == '__main__':
  solve_2()

