def forwardEulerSystem( fun, interval, y0, N ) :
    #  FORWARDEULER Solve differential equations using the forward Euler method.
    #  [T, U] = FORWARDEULER( FUN, INTERVAL, Y0, N ), with INTERVAL = [T0, TF],
    #  integrates the system of differential equations y'=f(t, y) from time T0 
    #  to time TF, with initial condition Y0, using the forward Euler method on 
    #  an equispaced grid of N intervals. Function FUN(T, Y) must return
    #  a column vector corresponding to f(t, y). Each row in the solution 
    #  array U corresponds to a time returned in the column vector T.
    import numpy as np

    # time step
    h = ( interval[1] - interval[0] ) / N
  
    # time snapshots 
    t = np.linspace( interval[0], interval[1], N+1 )

    # initialize the solution vector
    u = np.zeros([y0.size,N+1])
    u[:,0] = y0

    # time loop (n=0,...,n, but array indeces in Matlab start at 1)
    for n in range(N) :
        u[:,n+1] = u[:,n] + h * fun( t[n], u[:,n] )
  
    return t, u


def backwardEulerSystem( A, b, interval, y0, N ) :
    #  BACKWARD Solve differential equations using the forward Euler method.
    #  [T, U] = FORWARDEULER( A, b, INTERVAL, Y0, N ), with INTERVAL = [T0, TF],
    #  integrates the system of differential equations y'=f(t, y) from time T0 
    #  to time TF, with initial condition Y0, using the forward Euler method on 
    #  an equispaced grid of N intervals. 
    #  This version is limited to f(t,x) = A(t)x + b(t).
    import numpy as np

    d  = y0.size;
    # time step
    h = ( interval[1] - interval[0] ) / N
  
    # time snapshots 
    t = np.linspace( interval[0], interval[1], N+1 )

    # initialize the solution vector
    u = np.zeros([d,N+1])
    u[:,0] = y0

    # time loop (n=0,...,n, but array indeces in Matlab start at 1)
    for n in range(N) :
        B = (np.eye(d,d) - h*A(t[n+1]));
        u[:,n+1] = np.linalg.solve(B, u[:,n] + h * b( t[n+1]) );
  
    return t, u
