clearvars; 
close all

% Soil Parameters (loam) --------------------------------------------------
% Keep the units reported here
n       = 0.45;              % -         Soil Porosity
Zr      = ...                % cm        Rooting depth
Ks      = ...                % cm/day    Saturated hydraulic conductivity
beta    = ...                % -         Parameter depending on soil type
sh      = ...                % -         Hygroscopic point         
sw      = ...                % -         Wilting point
sstar   = ...                % -         Incipient stomatal closure point
sfc     = ...                % -         Field capacity point
s_ti    = 0.75;              % -         Initial value of soil moisture 

Ew      = ...                % cm/day    Evaporation at sw 
Emax    = ...                % cm/day    Evaporation at sstar 

% Stochastic rainfall parameters ------------------------------------------
alpha   = ...                % cm/day    Mean depth of daily rainfall 
lambda  = ...                % 1/day     Frequency of rainfall occurrence (rainfall rate)

% Times -------------------------------------------------------------------
dt      = 0.1;               % day
Tmax    = 365;               % day       
times   = 0:dt:Tmax;         % day

%% Temporal evolution of soil moisture dynamics 
nZr     = n*Zr;
gamma   = alpha/nZr;              
eta     = Emax/nZr;  

for jj=1:31 % 31 realizations
    s0 = 0.75;
    for t=1:length(times)

        % Rainfall idealized as pulses (Poisson process)
        if (rand(1) > lambda*dt)  %no rainfall
            Ft = 0;
        else
            Ft = exprnd(gamma); %random extraction from a Poisson process with mean gamma
        end
        
        % Losses
        if s_ti < sh
            LOSS(t) = ...
        elseif s_ti >= sh && s_ti <= sw
            LOSS(t) = ...
        elseif s_ti > sw && s_ti <= sfc
            LOSS(t) = ...
        elseif s_ti > sstar && s_ti <= sfc
            LOSS(t) = ...
        else
            m       = ...
            LOSS(t) = ...
        end

        s_ti = s0 -LOSS(t)*dt + Ft; %soil water balance equation
        xt(jj,t) = s_ti;
        s0 = s_ti;
    end

    plot(times,xt(jj,:),'Color',[.65 .65 .65])
    hold on
end

%% Choose one realization and plot it on top of the other, with a thicker black line
hold on
...

%% Plot the probability density function
S = ...  % eliminate the spin-up (around the first ~10 days)

figure; histogram(S,'Normalization','pdf')
%or
pdf = ksdensity(S,'Function','pdf');
axx = linspace(min(S), max(S));
figure;
plot(axx,pdf)

...