clear all
close all

% Global vatriables are needed for the balanceDRYDOWN function. 
% Please adopt the following names for the global variables

global rho rhow lambdaw cp gsrp psiS gsa ga qi qABL tABL PAR i

% rho      kg/m3     : Air density
% rhow     kg/m3     : Water density
% lambdaw  J/kg      : Latent heat of vaporization of water
% cp       J/kg/K    : Specific heat of air
% gsrp     m/MPa/s   : Soil-root-plant conductance
% psiS     MPa       : Air entry suction point (Brooks Corey)
% gsa      m/s       : Stomata-atmosphere conductance. gsa = gs*ga/(gs+ga)
% ga       m/s       : Atmospheric conductance
% qi       kg/kg     : Humidity inside the leaf
% qABL     kg/kg     : Humidity outside the leaf
% tABL     K         : Temperature of the ABL
% PAR      W/m2      : Absorbed photon irradiance
% i                  : Index of the for loop (temporal loop)


%%%%%%% Write down all the constants and parameters of Tab.s 7-11 %%%%%%%%%
%%%%%%%       Important: make sure that units are consistent      %%%%%%%%%

% General constants -------------------------------------------------------
g        = 9.8;          % m/s^2         Gravitational acceleration
rhow     = 998;          % kg/m^3        Water density
k        = 0.41;         % -             Von Karman Constant
R        = 8.314;        % J/(mol K)     Universal Gas Constant
nuv      = 18/1000/rhow; % m3/mol        Molecular volume of liquid water
rho      = 1.2;          % kg/m3         Air density
cp       = 1012;         % J/kg//K       Specific heat of air
cc       = 3e+8;         % m/s           Speed of light
NA       = 6.022e23;     % 1/mol         Avogadro's constant
h_planck = 6.63e-34;     % J s           Planck's constant
lambdaw  = 2.26e6;       % J/kg          Latent heat of vaporization of water

% Soil Parameters ---------------------------------------------------------
psiss   = ...                % MPa       Air entry suction point (Brooks Corey)
b       = ...                % -         Parameter related to pore size distribution (Brooks Corey)
Ks      = ...                % m/s       Saturated hydraulic conductivity
n       = ...                % -         Soil Porosity
Ew      = ...                % m/s       Soil evaporation 
sw      = 0.2;               % -         Wilting point
sh      = 0.05;              % -         Hygroscopic point         

% Plant parameters --------------------------------------------------------
Zr      = ...           % m             Rooting depth
LAI     = ...           % -             Leaf area per unit ground area
RAIW    = ...           % -             Root Area Index under well watered conditions
a       = 8;            % -             Parameter accounting for root growth
droot   = ...           % m             Root diameter 
hc      = ...           % m             Canopy height
c1      = 2;            % -             Parameter of Vulnerability Curve
d1      = 2;            % MPa           Parameter of Vulnerability Curve
gpmax   = 11.7*1e-6;    % m/(MPa*s)     Maximum Plant Conductance
k1      = ...           % m^2/W         Parameter f_phi
k2      = ...           % 1/K^2         Parameter f_Ta
Topt    = 298;          % K             Parameter f_Ta
Dx      = 1250/1e6;     % MPa           Parameter f_D 
psi_l0  = -4.5;         % MPa           Parameter f_psil
psi_l1  = -0.05;        % MPa           Parameter f_phil
gsmax   = 25*1e-3;      % m/s           Maximum stomatal conductance 

% Atmospheric parameters ---------------------------------------------------
p0      = ...           % MPa           Atmospheric pressure
ca      = ...           % mol/mol       Atmospheric CO2 concentration 
ci      = ...           % mol/mol       CO2 concentration in the stomatal pore
Uw      = ...           % m/s           Wind speed
d       = 0.75*hc;      % m             Displacement height
epsil   = d/10;         % m             Momentum roughness height   
epsilq  = 0.2*epsil;    % m             Water vapor roughness height
ga      = ...           % m/s           Aerodynamic conductance
t0      = ...           % s             Parameter for leaf available energy
delta   = ...           % s             Day length


% Carbon assimilation parameters ------------------------------------------
Hkc     = ...           % J/mol         Activation energy for Kc          
Hko     = ...           % J/mol         Activation energy for Ko          
Hvv     = 116300;       % J/mol         Activation energy for Vcmax       
Hdv     = ...           % J/mol         Deactivation energy for Vcmax     
Hvj     = ...           % J/mol         Activation energy for Jmax        
Hdj     = ...           % J/mol         Deactivation energy for Jmax      
Jmax0   = 75;           % mumol_e/m2/s  CO2 Assimilation parameter       
Kc0     = ...           % mumol/mol     Michaelis constant for CO2 at T0  
Ko0     = ...           % mmol/mol      Michaelis constant for O2 at T0   
oi      = 0.209;        % mol/mol       Oxygen concentration              
T0      = ...           % K             Reference temperature            
Sv      = 650;          % J/mol         Entropy term                     
Vcmax0  = ...           % mumol/m2/s    CO2 Assimilation parameter       
gamma0  = ...           % mumol/mol     CO2 compensation point at T0     
gamma1  = ...           % 1/K           Parameter compensation point      
gamma2  = ...           % 1/K2          Parameter compensation point     
wave    = ...           % m             Average wavelength of PAR 
k_2     = 0.20;         % mol_e/mol_ph  Quantum yield of photosynthesis 



%%%%%%%%%%%%%%%%% Create a time vector over 10-day period %%%%%%%%%%%%%%%%%
% Create the time vector 
days_tot = 10;                          % days   Length of the dry-down  
Deltat   = 1200;                        % s      Timestep (20 min)
day_1    = Deltat:Deltat:86400;         % s      Single day
time     = repmat(day_1, 1, days_tot);  % s


%% Initialization 
s        = 0.5;                                % -
hABL     = 50;                                 % m
tABL     = 283;                                % K
qABL     = 0.0008;                             % kg/kg
PAR      = 0;                                  % W/m2  Absorbed photon irradiance 
PHI      = 0;                                  % W/m2  Incoming radiation
if (s>sw)
    E    = Ew;
elseif(s<sh)
    E    = 0;
else
    E    = Ew/(sw-sh)*s-Ew*sh/(sw-sh);
end
psiS     = psiss*s^(-b);                       % MPa
LEAK     = 0;                                  % m/s
EV       = 0;                                  % m/s
tFA      = 4.78/1000*hABL+293.6;               % K
qFA      = -0.00285/1000*hABL+0.01166;         % kg/kg
RAI      = RAIW*s^(-a);                        % - 
gsr      = LEAK*sqrt(RAI)/(pi*g*rhow*Zr)*1e6;  % m/(s*MPa)
PSIleaf  = psiS;                               % m 
fTj      = 1-k2*(tABL-Topt).^2;                % -
fLIGHTj  = 1-exp(-k1*PHI);                     % - 
if (PSIleaf<psi_l0)                            % -
    fPSIj= 0;
elseif (psi_l0 <= PSIleaf) && ( PSIleaf <= psi_l1)
    fPSIj= (PSIleaf-psi_l0)./(psi_l1-psi_l0);
elseif (PSIleaf > psi_l1)
    fPSIj= 1;
end
esat     = 611.71*exp((17.27*(tABL-273.15))/(tABL-273.15+237.3))*1e-6; % MPa
VPD      = esat-qABL*p0/0.622;                 % MPa
fVPDj    = 1/(1+VPD/Dx);                       % -
gp       = gpmax*exp(-(-PSIleaf/d1).^c1);      % m/(MPa s)
gsrp     = LAI*gp*gsr/(LAI*gp+gsr(1));         % m/(MPa s)
tLEAF    = tABL;                               % K
qi       = (611.71*exp((17.27*(tLEAF-273.15))/(tLEAF-273.15+237.3)))*1e-6*exp(nuv*(PSIleaf-g*rhow*hc/1e6)/R/tLEAF*1e6)*0.622/p0; % kg/kg                                                                               
gs       = gsmax*fPSIj*fVPDj*fLIGHTj*fTj;      % m/s
gsa      = gs*ga/(ga+gs);                      % m/s
H        = 0;
Vcmax    = 0;
Kc       = 0;
Ko       = 0;
Comp     = 0;
A_C      = 0;
Jmax     = 0;
J        = 0;
A_q      = 0;
Anet     = 0;   


%% Temporal evolution
for i=2:length(time)
        
    if (time(i)<=6*3600) || (time(i)>=18*3600) %<6h or >18h, i.e., nighttime
        PAR(i) = 0;
    else
        PAR(i) = (4*500/(delta^2))*(-(time(i))^2+(delta+2*t0)*time(i)-t0*(t0+delta)); % W/m2
    end
    PHI(i) = PAR(i);
    J_PHI(i)= PAR(i)*wave*k_2/(NA*cc*h_planck)*1e6;       
    fLIGHTj(i)= 1-exp(-k1*PHI(i));  
        
    if ((time(i)<=6*3600) || (time(i)>=18*3600)) %<6h or >18h, i.e., nighttime
        s(i)        = s(i-1);
        E(i)        = E(i-1);
        RAI(i)      = ...( f(s(i)) )
        psiS(i)     = ...( f(s(i)) )
        LEAK(i)     = 0;
        gsr(i)      = gsr(1);  
        H(i)        = H(1);
        EV(i)       = 0;
        hABL(i)     = hABL(1);
        tFA(i)      = tFA(1);
        qFA(i)      = qFA(1);
        tABL(i)     = tABL(1);
        qABL(i)     = qABL(1);
        qi(i)       = qi(1); 
        gp(i)       = gp(1);
        gsrp(i)     = gsrp(1);
        esat(i)     = esat(1);       
        VPD(i)      = VPD(1);
        fVPDj(i)    = fVPDj(1); 
        fTj(i)      = fTj(1);
        gs(i)       = gs(1);
        gsa(i)      = gsa(1);
      
        PSIleaf(i)  = psiS(i);
        tLEAF(i)    = tLEAF(1);

        if (PSIleaf(i-1) < psi_l0)
            fPSIj(i)  =  0;
        elseif (PSIleaf(i-1) > psi_l1)
            fPSIj(i)  = 1;
        else
            fPSIj(i)  =  (PSIleaf(i-1)-psi_l0)/(psi_l1-psi_l0);
        end 
        
        Vcmax(i)    = 0;
        Kc(i)       = 0;
        Ko(i)       = 0;
        Comp(i)     = 0;
        A_C(i)      = 0;
        Jmax(i)     = 0;
        J(i)        = 0;
        A_q(i)      = 0;
        Anet(i)     = 0;   
        
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%&    
    else   % daytime
        if (PSIleaf(i-1) < psi_l0)
            fPSIj(i)  =  0;
        elseif (PSIleaf(i-1) > psi_l1)
            fPSIj(i)  = 1;
        else
            fPSIj(i)  =  (PSIleaf(i-1)-psi_l0)/(psi_l1-psi_l0);
        end 

        s(i)      = ... [discretized equation for s]

        if (s(i)>sw)
            E(i)  = Ew;
        elseif(s(i)<sh)
            E(i)  = 0;
        else
            % E(i) = ... f(s(i)) % Piecewise approx. function
        end
        
        psiS(i)   = ... f(s(i))
        LEAK(i)   = ... f(s(i))
        RAI(i)    = ... f(s(i))                  
        gsr(i)    = LEAK(i)*sqrt(RAI(i))/(pi*g*rhow*Zr)*1e6;  % Provided equation
        H(i)      = ... f(tLEAF(i-1)-tABL(i-1)) 
        EV(i)     = ... f(qi(i-1)-qABL(i-1)) % m/s  
        hABL(i)   = ... [discretized hABL] 
        tFA(i)    = ...  
        qFA(i)    = ...
        tABL(i)   = ... [discretized tABL] 
        qABL(i)   = ... [discretized qABL]       
        qi(i)     = ...f(tLEAF(i-1),PSIleaf(i-1)) %kg/kg                                                                                     
        gp(i)     = ...f(PSIleaf(i-1))  
        gsrp(i)   = ...f(gp(i),gsr(i))    
        esat(i)   = ...f(tABL(i)) % MPa
        VPD(i)    = esat(i)-qABL(i)*p0/0.622;                 % Provided equation
        fVPDj(i)  = ...
        fTj(i)    = ...
        gs(i)     = ...f(fPSIj(i),fVPDj(i),fLIGHTj(i),fTj(i))
        gsa(i)    = ...f(gs(i),ga);

        %% Solve the system of equations for PSIleaf and tLEAF 
        options   = optimset('Display','iter');
        X0        = [PSIleaf(i-1) tLEAF(i-1)];
        [out,fval]= fsolve(@balanceDRYDOWN,X0,options) ;
        PSIleaf(i)= out(1);
        tLEAF(i)  = out(2);

        %% Model Net Assimilation - Farquhar et al. (1980) model
        Vcmax(i)  = ...                    % mumol/m2/s   Maximum carboxylation rate
        Kc(i)     = ...                    % mumol/m      Michaelis-Menten coefficients
        Ko(i)     = ...                    % mumol/m      Michaelis-Menten coefficients
        Comp(i)   = ...                    % mumol/m      Equilibrium CO2 compensation point 
        A_C(i)    = ...                    % mumol/m2/s   CO2 limitation due to rubisco activity
        Jmax(i)   = ...                    % mumol/m2/s   Maximum electron transport
        J(i)      = min(Jmax(i),J_PHI(i)); % mol/m2/s     Electron transport rate
        A_q(i)    = ...
        Anet(i)   = ...
       
    end
end
    
%% Plot the results
figure(1)
%% Temporal evolution of s
subplot(3,2,1)
plot(...,...,'k','LineWidth',1.5) 
xlabel('t[d]')                     % report time in [days] 
ylabel('s [-]')
xlim([...])                        % adjust x axis for better visibility

%% Temporal evolution of gs 
subplot(3,2,2)
plot(...,...,'-.k','LineWidth',1.5) 
xlabel('t[d]')
ylabel('g_s [mm/s]')               % report gs in [mm/s]
xlim([...])

%% Temporal evolution of PSIleaf and psiS
subplot(3,2,3)
plot(...,...,'k','LineWidth',1.5)
hold on
plot(... , ...,'--k','LineWidth',1.5)
xlabel('t[d]')
ylabel('\psi [MPa]')               % make sure that they are in [MPa]
legend('\psi_L','\psi_S')
xlim([...])

%% Temporal evolution of EV
subplot(3,2,4)
plot(...,...,'k','LineWidth',1.5)
xlabel('t[d]')
ylabel('E [mm/day]')               % report EV in [mm/day]
xlim([...])

%% Temporal evolution of Anet, A_q and A_C
subplot(3,2,5)
plot(...,...,'k','LineWidth',1.5) % report Anet in [mol/(m^2 d)]
hold on
plot(...,A_q [mol/(m^2 d)],':k','LineWidth',1.5)
hold on
plot(...,A_C [mol/(m^2 d)],'--k','LineWidth',1.5)
xlabel('t[d]')
ylabel('A_n [mol/(m^2 d)]]')
%legend...
xlim([...])

%% Temporal evolution of hABL
subplot(3,2,6)
plot(...,hABL,'k','LineWidth',1.5)
xlabel('t[d]')
ylabel('h_ABL [m]')
xlim(...)

