/*
Author: Sofia Ermida (sofia.ermida@ipma.pt; @ermida_sofia)

This code is free and open. 
By using this code and any data derived with it, 
you agree to cite the following reference 
in any publications derived from them:
Ermida, S.L., Soares, P., Mantas, V., Göttsche, F.-M., Trigo, I.F., 2020. 
    Google Earth Engine open-source code for Land Surface Temperature estimation from the Landsat series.
    Remote Sensing, 12 (9), 1471; https://doi.org/10.3390/rs12091471

Example 1:
  This example shows how to compute Landsat LST from Landsat-8 over Coimbra
  This corresponds to the example images shown in Ermida et al. (2020)
    
*/

///////////////////////////// ATTENTION //////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
// As off 11.07.2022 a new version of the code is released:
//      - update to use collection 2 data
//      - emissivities of water and snow surfaces are now prescribed 
// 
// the previous version of the code will still be available; the replaced code
// is commented
//
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

// TEST for america

var state = 'Geneva'

var satellite = 'L8';
var date_start = '2022-05-01';  // Expand the date range to include more images for compositing
var date_end = '2022-08-31';
var use_ndvi = true;

var LandsatLST = require('users/sofiaermida/landsat_smw_lst:modules/Landsat_LST.js')

function calculate_LST(state){
  var aoi = ee.FeatureCollection("FAO/GAUL_SIMPLIFIED_500m/2015/level1").filter(ee.Filter.eq('ADM1_NAME', state)).geometry();
  var LandsatColl = LandsatLST.collection(satellite, date_start, date_end, aoi, use_ndvi);
  var compositeImage = LandsatColl.median();
  var clippedImage = compositeImage.clip(aoi);
  var cmap1 = ['blue', 'cyan', 'green', 'yellow', 'red'];

  Export.image.toDrive({
    image: clippedImage.select('LST'),
    description: 'LST_Arusha_May_2022'+'_'+state,
    scale: 30,
    region: aoi,
    fileFormat: 'GeoTIFF',
  });
}

calculate_LST(state);



// link to the code that computes the Landsat LST




// Define the AOI using the GAUL dataset for Arusha (as an example)
var aoi = ee.FeatureCollection("FAO/GAUL_SIMPLIFIED_500m/2015/level1")
            .filter(ee.Filter.eq('ADM1_NAME', 'Arusha'))
            .geometry();

var satellite = 'L8';
var date_start = '2022-01-01';  // Expand the date range to include more images for compositing
var date_end = '2022-05-31';
var use_ndvi = true;

// Get Landsat collection with added variables: NDVI, FVC, TPW, EM, LST
var LandsatColl = LandsatLST.collection(satellite, date_start, date_end, aoi, use_ndvi);

// Apply median compositing to reduce the impact of clouds
var compositeImage = LandsatColl.median();
print(compositeImage);

// Clip the composite image to your AOI
var clippedImage = compositeImage.clip(aoi);

// Visualization parameters
var cmap1 = ['blue', 'cyan', 'green', 'yellow', 'red'];
var cmap2 = ['F2F2F2','EFC2B3','ECB176','E9BD3A','E6E600','63C600','00A600']; 

// Center the map on the AOI
Map.centerObject(aoi, 10);

// Add layers to the map for visualization
Map.addLayer(clippedImage.select('TPW'), {min:0.0, max:60.0, palette:cmap1}, 'TCWV');
Map.addLayer(clippedImage.select('TPWpos'), {min:0.0, max:9.0, palette:cmap1}, 'TCWVpos');
Map.addLayer(clippedImage.select('FVC'), {min:0.0, max:1.0, palette:cmap2}, 'FVC');
Map.addLayer(clippedImage.select('EM'), {min:0.9, max:1.0, palette:cmap1}, 'Emissivity');
Map.addLayer(clippedImage.select('B10'), {min:290, max:320, palette:cmap1}, 'TIR BT');
Map.addLayer(clippedImage.select('LST'), {min:290, max:320, palette:cmap1}, 'LST');
Map.addLayer(clippedImage.multiply(0.0000275).add(-0.2), {bands: ['SR_B4', 'SR_B3', 'SR_B2'], min:0, max:0.3}, 'RGB');

// Uncomment the code below to export an image band to your drive
/*
Export.image.toDrive({
  image: clippedImage.select('LST'),
  description: 'LST_Arusha_May_2022',
  scale: 30,
  region: aoi,
  fileFormat: 'GeoTIFF',
});
*/